package com.fishsemi.sdk.aircontrol;

import android.os.Handler;
import android.os.RemoteException;
import android.util.Log;
import com.fishsemi.sdk.aircontrol.Types;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;

public class DataStream {
    private static int[] BindPorts = {14551, 13551};
    private static int[] ConnectPorts = {14552, 13552};
    private static int MaxUartID = 2;
    private static final String TAG = "DataStream";
    private AirController mAirController = null;
    private DatagramSocket mDatagramSocket = null;
    private Handler mHandler = null;
    private DataStreamListener mListener = null;
    private int mUartId = 0;
    private boolean mWaitingResult = false;

    public DataStream(DataStreamListener dataStreamListener) {
        initDataStream(1, dataStreamListener);
    }

    public DataStream(int i, DataStreamListener dataStreamListener) {
        initDataStream(i, dataStreamListener);
    }

    public DataStream(AirController airController, int i, DataStreamListener dataStreamListener) {
        initDataStream(i, dataStreamListener);
        setAirController(airController);
    }

    public void start() {
        new Thread(new Runnable() {
            /* class com.fishsemi.sdk.aircontrol.DataStream.AnonymousClass1 */

            public void run() {
                if (DataStream.this.initConnection()) {
                    DataStream.this.receiveDataLoop();
                }
            }
        }, "DataReceive").start();
    }

    public void stop() {
        closeConnection();
    }

    public void setAirController(AirController airController) {
        this.mAirController = airController;
        this.mAirController.addDataStream(this);
    }

    public int getUartId() {
        return this.mUartId;
    }

    public void setUartId(int i) {
        if (this.mUartId != i) {
            this.mUartId = i;
            DatagramSocket datagramSocket = this.mDatagramSocket;
            if (datagramSocket != null && datagramSocket.isConnected()) {
                stop();
                start();
            }
            AirController airController = this.mAirController;
            if (airController != null) {
                airController.addDataStream(this);
            }
        }
    }

    public DataStreamListener getListener() {
        return this.mListener;
    }

    public void onSetUartBaudrateDone(Types.Result result) {
        if (this.mWaitingResult) {
            DataStreamListener dataStreamListener = this.mListener;
            if (dataStreamListener != null) {
                dataStreamListener.onSetUartBaudrateDone(result);
            }
            this.mWaitingResult = false;
        }
    }

    public int sendData(byte[] bArr, int i) {
        if (this.mDatagramSocket == null) {
            Log.e(TAG, "socket not ready, failed to send data");
            return 0;
        }
        DatagramPacket datagramPacket = new DatagramPacket(bArr, i);
        try {
            this.mDatagramSocket.send(datagramPacket);
            return datagramPacket.getLength();
        } catch (IOException e) {
            Log.e(TAG, "failed to send data " + e);
            return 0;
        }
    }

    public int getUartBaudrate() {
        AirController airController = this.mAirController;
        if (!(airController == null || airController.getAirControlService() == null)) {
            try {
                return this.mAirController.getAirControlService().getUartBaudrate(this.mUartId);
            } catch (RemoteException e) {
                e.printStackTrace();
            }
        }
        return 0;
    }

    public Types.RetValue setUartBaudrate(int i) {
        AirController airController = this.mAirController;
        if (airController == null || airController.getAirControlService() == null) {
            return Types.RetValue.NOT_READY;
        }
        try {
            int uartBaudrate = this.mAirController.getAirControlService().setUartBaudrate(this.mUartId, i);
            if (AirController.toRetValue(uartBaudrate) == Types.RetValue.OK) {
                this.mWaitingResult = true;
            }
            return AirController.toRetValue(uartBaudrate);
        } catch (RemoteException e) {
            e.printStackTrace();
            return Types.RetValue.REMOTE_ERROR;
        }
    }

    private void initDataStream(int i, DataStreamListener dataStreamListener) {
        if (i > MaxUartID || i < 1) {
            i = 1;
        }
        this.mUartId = i;
        this.mListener = dataStreamListener;
    }

    /* access modifiers changed from: private */
    /* access modifiers changed from: public */
    private boolean initConnection() {
        try {
            this.mDatagramSocket = new DatagramSocket(BindPorts[this.mUartId - 1]);
            this.mDatagramSocket.connect(InetAddress.getByName("127.0.0.1"), ConnectPorts[this.mUartId - 1]);
            Log.d(TAG, "data connection is created");
            return true;
        } catch (Exception e) {
            Log.e(TAG, "failed to init connection. " + e);
            return false;
        }
    }

    private void closeConnection() {
        DatagramSocket datagramSocket = this.mDatagramSocket;
        if (datagramSocket != null) {
            datagramSocket.close();
            Log.d(TAG, "data connection is close");
        }
    }

    /* access modifiers changed from: private */
    /* access modifiers changed from: public */
    private void receiveDataLoop() {
        DatagramPacket datagramPacket = new DatagramPacket(new byte[256], 256);
        while (true) {
            DatagramSocket datagramSocket = this.mDatagramSocket;
            if (datagramSocket != null && datagramSocket.isConnected()) {
                try {
                    this.mDatagramSocket.receive(datagramPacket);
                    if (this.mListener != null && datagramPacket.getLength() > 0) {
                        if (this.mHandler == null) {
                            this.mHandler = new Handler(this.mAirController.getContext().getMainLooper());
                        }
                        final byte[] bArr = new byte[datagramPacket.getLength()];
                        System.arraycopy(datagramPacket.getData(), 0, bArr, 0, bArr.length);
                        this.mHandler.post(new Runnable() {
                            /* class com.fishsemi.sdk.aircontrol.DataStream.AnonymousClass2 */

                            public void run() {
                                DataStream.this.mListener.onDataReceived(bArr);
                            }
                        });
                    }
                } catch (IOException e) {
                    Log.e(TAG, "stop to received data, " + e);
                    return;
                }
            } else {
                return;
            }
        }
    }
}
