package io.grpc;

import com.google.common.io.ByteStreams;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;

public final class TlsServerCredentials extends ServerCredentials {
    private final byte[] certificateChain;
    private final boolean fakeFeature;
    private final byte[] privateKey;
    private final String privateKeyPassword;

    public enum Feature {
        FAKE
    }

    public static ServerCredentials create(File file, File file2) throws IOException {
        return newBuilder().keyManager(file, file2).build();
    }

    public static ServerCredentials create(InputStream inputStream, InputStream inputStream2) throws IOException {
        return newBuilder().keyManager(inputStream, inputStream2).build();
    }

    TlsServerCredentials(Builder builder) {
        this.fakeFeature = builder.fakeFeature;
        this.certificateChain = builder.certificateChain;
        this.privateKey = builder.privateKey;
        this.privateKeyPassword = builder.privateKeyPassword;
    }

    public byte[] getCertificateChain() {
        byte[] bArr = this.certificateChain;
        return Arrays.copyOf(bArr, bArr.length);
    }

    public byte[] getPrivateKey() {
        byte[] bArr = this.privateKey;
        return Arrays.copyOf(bArr, bArr.length);
    }

    public String getPrivateKeyPassword() {
        return this.privateKeyPassword;
    }

    public Set<Feature> incomprehensible(Set<Feature> set) {
        EnumSet noneOf = EnumSet.noneOf(Feature.class);
        if (this.fakeFeature) {
            requiredFeature(set, noneOf, Feature.FAKE);
        }
        return Collections.unmodifiableSet(noneOf);
    }

    private static void requiredFeature(Set<Feature> set, Set<Feature> set2, Feature feature) {
        if (!set.contains(feature)) {
            set2.add(feature);
        }
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static final class Builder {
        private byte[] certificateChain;
        private boolean fakeFeature;
        private byte[] privateKey;
        private String privateKeyPassword;

        private Builder() {
        }

        public Builder requireFakeFeature() {
            this.fakeFeature = true;
            return this;
        }

        public Builder keyManager(File file, File file2) throws IOException {
            return keyManager(file, file2, (String) null);
        }

        public Builder keyManager(File file, File file2, String str) throws IOException {
            FileInputStream fileInputStream = new FileInputStream(file);
            try {
                fileInputStream = new FileInputStream(file2);
                try {
                    Builder keyManager = keyManager(fileInputStream, fileInputStream, str);
                    fileInputStream.close();
                    return keyManager;
                } finally {
                    fileInputStream.close();
                }
            } finally {
                fileInputStream.close();
            }
        }

        public Builder keyManager(InputStream inputStream, InputStream inputStream2) throws IOException {
            return keyManager(inputStream, inputStream2, (String) null);
        }

        public Builder keyManager(InputStream inputStream, InputStream inputStream2, String str) throws IOException {
            byte[] byteArray = ByteStreams.toByteArray(inputStream);
            byte[] byteArray2 = ByteStreams.toByteArray(inputStream2);
            this.certificateChain = byteArray;
            this.privateKey = byteArray2;
            this.privateKeyPassword = str;
            return this;
        }

        public ServerCredentials build() {
            if (this.certificateChain != null) {
                return new TlsServerCredentials(this);
            }
            throw new IllegalStateException("A key manager is required");
        }
    }
}
