package com.google.common.util.concurrent;

import com.google.common.base.Function;
import com.google.common.base.Functions;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.Futures;
import com.google.errorprone.annotations.DoNotMock;
import java.io.Closeable;
import java.io.IOException;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.checkerframework.checker.nullness.compatqual.NullableDecl;

@DoNotMock("Use ClosingFuture.from(Futures.immediate*Future)")
public final class ClosingFuture<V> {
    private static final Logger logger = Logger.getLogger(ClosingFuture.class.getName());
    private final CloseableList closeables;
    private final FluentFuture<V> future;
    private final AtomicReference<State> state;

    public interface AsyncClosingFunction<T, U> {
        ClosingFuture<U> apply(DeferredCloser deferredCloser, @NullableDecl T t) throws Exception;
    }

    public interface ClosingCallable<V> {
        @NullableDecl
        V call(DeferredCloser deferredCloser) throws Exception;
    }

    public interface ClosingFunction<T, U> {
        @NullableDecl
        U apply(DeferredCloser deferredCloser, @NullableDecl T t) throws Exception;
    }

    /* access modifiers changed from: package-private */
    public enum State {
        OPEN,
        SUBSUMED,
        WILL_CLOSE,
        CLOSING,
        CLOSED,
        WILL_CREATE_VALUE_AND_CLOSER
    }

    public interface ValueAndCloserConsumer<V> {
        void accept(ValueAndCloser<V> valueAndCloser);
    }

    public static final class DeferredCloser {
        private final CloseableList list;

        DeferredCloser(CloseableList closeableList) {
            this.list = closeableList;
        }

        @NullableDecl
        public <C extends Closeable> C eventuallyClose(@NullableDecl C c, Executor executor) {
            Preconditions.checkNotNull(executor);
            if (c != null) {
                this.list.add(c, executor);
            }
            return c;
        }
    }

    public static final class ValueAndCloser<V> {
        private final ClosingFuture<? extends V> closingFuture;

        ValueAndCloser(ClosingFuture<? extends V> closingFuture2) {
            this.closingFuture = (ClosingFuture) Preconditions.checkNotNull(closingFuture2);
        }

        @NullableDecl
        public V get() throws ExecutionException {
            return (V) Futures.getDone(((ClosingFuture) this.closingFuture).future);
        }

        public void closeAsync() {
            this.closingFuture.close();
        }
    }

    public static <V> ClosingFuture<V> submit(ClosingCallable<V> closingCallable, Executor executor) {
        return new ClosingFuture<>(closingCallable, executor);
    }

    public static <V> ClosingFuture<V> from(ListenableFuture<V> listenableFuture) {
        return new ClosingFuture<>(listenableFuture);
    }

    @Deprecated
    public static <C extends Closeable> ClosingFuture<C> eventuallyClosing(ListenableFuture<C> listenableFuture, final Executor executor) {
        Preconditions.checkNotNull(executor);
        ClosingFuture<C> closingFuture = new ClosingFuture<>(Futures.nonCancellationPropagating(listenableFuture));
        Futures.addCallback(listenableFuture, new FutureCallback<Closeable>() {
            /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass1 */

            @Override // com.google.common.util.concurrent.FutureCallback
            public void onFailure(Throwable th) {
            }

            public void onSuccess(@NullableDecl Closeable closeable) {
                ClosingFuture.this.closeables.closer.eventuallyClose(closeable, executor);
            }
        }, MoreExecutors.directExecutor());
        return closingFuture;
    }

    public static Combiner whenAllComplete(Iterable<? extends ClosingFuture<?>> iterable) {
        return new Combiner(false, iterable);
    }

    public static Combiner whenAllComplete(ClosingFuture<?> closingFuture, ClosingFuture<?>... closingFutureArr) {
        return whenAllComplete(Lists.asList(closingFuture, closingFutureArr));
    }

    public static Combiner whenAllSucceed(Iterable<? extends ClosingFuture<?>> iterable) {
        return new Combiner(true, iterable);
    }

    public static <V1, V2> Combiner2<V1, V2> whenAllSucceed(ClosingFuture<V1> closingFuture, ClosingFuture<V2> closingFuture2) {
        return new Combiner2<>(closingFuture2);
    }

    public static <V1, V2, V3> Combiner3<V1, V2, V3> whenAllSucceed(ClosingFuture<V1> closingFuture, ClosingFuture<V2> closingFuture2, ClosingFuture<V3> closingFuture3) {
        return new Combiner3<>(closingFuture2, closingFuture3);
    }

    public static <V1, V2, V3, V4> Combiner4<V1, V2, V3, V4> whenAllSucceed(ClosingFuture<V1> closingFuture, ClosingFuture<V2> closingFuture2, ClosingFuture<V3> closingFuture3, ClosingFuture<V4> closingFuture4) {
        return new Combiner4<>(closingFuture2, closingFuture3, closingFuture4);
    }

    public static <V1, V2, V3, V4, V5> Combiner5<V1, V2, V3, V4, V5> whenAllSucceed(ClosingFuture<V1> closingFuture, ClosingFuture<V2> closingFuture2, ClosingFuture<V3> closingFuture3, ClosingFuture<V4> closingFuture4, ClosingFuture<V5> closingFuture5) {
        return new Combiner5<>(closingFuture2, closingFuture3, closingFuture4, closingFuture5);
    }

    public static Combiner whenAllSucceed(ClosingFuture<?> closingFuture, ClosingFuture<?> closingFuture2, ClosingFuture<?> closingFuture3, ClosingFuture<?> closingFuture4, ClosingFuture<?> closingFuture5, ClosingFuture<?> closingFuture6, ClosingFuture<?>... closingFutureArr) {
        return whenAllSucceed(FluentIterable.of(closingFuture, closingFuture2, closingFuture3, closingFuture4, closingFuture5, closingFuture6).append(closingFutureArr));
    }

    private ClosingFuture(ListenableFuture<V> listenableFuture) {
        this.state = new AtomicReference<>(State.OPEN);
        this.closeables = new CloseableList();
        this.future = FluentFuture.from(listenableFuture);
    }

    private ClosingFuture(final ClosingCallable<V> closingCallable, Executor executor) {
        this.state = new AtomicReference<>(State.OPEN);
        this.closeables = new CloseableList();
        Preconditions.checkNotNull(closingCallable);
        TrustedListenableFutureTask create = TrustedListenableFutureTask.create(new Callable<V>() {
            /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass2 */

            @Override // java.util.concurrent.Callable
            public V call() throws Exception {
                return (V) closingCallable.call(ClosingFuture.this.closeables.closer);
            }

            public String toString() {
                return closingCallable.toString();
            }
        });
        executor.execute(create);
        this.future = create;
    }

    public ListenableFuture<?> statusFuture() {
        return Futures.nonCancellationPropagating(this.future.transform(Functions.constant(null), MoreExecutors.directExecutor()));
    }

    public <U> ClosingFuture<U> transform(final ClosingFunction<? super V, U> closingFunction, Executor executor) {
        Preconditions.checkNotNull(closingFunction);
        return derive(this.future.transformAsync(new AsyncFunction<V, U>() {
            /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass3 */

            @Override // com.google.common.util.concurrent.AsyncFunction
            public ListenableFuture<U> apply(V v) throws Exception {
                return ClosingFuture.this.closeables.applyClosingFunction(closingFunction, v);
            }

            public String toString() {
                return closingFunction.toString();
            }
        }, executor));
    }

    public <U> ClosingFuture<U> transformAsync(final AsyncClosingFunction<? super V, U> asyncClosingFunction, Executor executor) {
        Preconditions.checkNotNull(asyncClosingFunction);
        return derive(this.future.transformAsync(new AsyncFunction<V, U>() {
            /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass4 */

            @Override // com.google.common.util.concurrent.AsyncFunction
            public ListenableFuture<U> apply(V v) throws Exception {
                return ClosingFuture.this.closeables.applyAsyncClosingFunction(asyncClosingFunction, v);
            }

            public String toString() {
                return asyncClosingFunction.toString();
            }
        }, executor));
    }

    public static <V, U> AsyncClosingFunction<V, U> withoutCloser(final AsyncFunction<V, U> asyncFunction) {
        Preconditions.checkNotNull(asyncFunction);
        return new AsyncClosingFunction<V, U>() {
            /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass5 */

            @Override // com.google.common.util.concurrent.ClosingFuture.AsyncClosingFunction
            public ClosingFuture<U> apply(DeferredCloser deferredCloser, V v) throws Exception {
                return ClosingFuture.from(AsyncFunction.this.apply(v));
            }
        };
    }

    /* JADX DEBUG: Multi-variable search result rejected for r2v0, resolved type: com.google.common.util.concurrent.ClosingFuture$ClosingFunction<? super X extends java.lang.Throwable, ? extends V> */
    /* JADX WARN: Multi-variable type inference failed */
    public <X extends Throwable> ClosingFuture<V> catching(Class<X> cls, ClosingFunction<? super X, ? extends V> closingFunction, Executor executor) {
        return catchingMoreGeneric(cls, closingFunction, executor);
    }

    /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.FluentFuture<V>, com.google.common.util.concurrent.FluentFuture<U> */
    /* JADX DEBUG: Type inference failed for r2v2. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<U>, com.google.common.util.concurrent.ClosingFuture<V> */
    private <X extends Throwable, W extends V> ClosingFuture<V> catchingMoreGeneric(Class<X> cls, final ClosingFunction<? super X, W> closingFunction, Executor executor) {
        Preconditions.checkNotNull(closingFunction);
        return (ClosingFuture<U>) derive((FluentFuture<V>) this.future.catchingAsync(cls, new AsyncFunction<X, W>() {
            /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass6 */

            public ListenableFuture<W> apply(X x) throws Exception {
                return ClosingFuture.this.closeables.applyClosingFunction(closingFunction, x);
            }

            public String toString() {
                return closingFunction.toString();
            }
        }, executor));
    }

    /* JADX DEBUG: Multi-variable search result rejected for r2v0, resolved type: com.google.common.util.concurrent.ClosingFuture$AsyncClosingFunction<? super X extends java.lang.Throwable, ? extends V> */
    /* JADX WARN: Multi-variable type inference failed */
    public <X extends Throwable> ClosingFuture<V> catchingAsync(Class<X> cls, AsyncClosingFunction<? super X, ? extends V> asyncClosingFunction, Executor executor) {
        return catchingAsyncMoreGeneric(cls, asyncClosingFunction, executor);
    }

    /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.FluentFuture<V>, com.google.common.util.concurrent.FluentFuture<U> */
    /* JADX DEBUG: Type inference failed for r2v2. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<U>, com.google.common.util.concurrent.ClosingFuture<V> */
    private <X extends Throwable, W extends V> ClosingFuture<V> catchingAsyncMoreGeneric(Class<X> cls, final AsyncClosingFunction<? super X, W> asyncClosingFunction, Executor executor) {
        Preconditions.checkNotNull(asyncClosingFunction);
        return (ClosingFuture<U>) derive((FluentFuture<V>) this.future.catchingAsync(cls, new AsyncFunction<X, W>() {
            /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass7 */

            public ListenableFuture<W> apply(X x) throws Exception {
                return ClosingFuture.this.closeables.applyAsyncClosingFunction(asyncClosingFunction, x);
            }

            public String toString() {
                return asyncClosingFunction.toString();
            }
        }, executor));
    }

    public FluentFuture<V> finishToFuture() {
        if (compareAndUpdateState(State.OPEN, State.WILL_CLOSE)) {
            logger.log(Level.FINER, "will close {0}", this);
            this.future.addListener(new Runnable() {
                /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass8 */

                public void run() {
                    ClosingFuture.this.checkAndUpdateState(State.WILL_CLOSE, State.CLOSING);
                    ClosingFuture.this.close();
                    ClosingFuture.this.checkAndUpdateState(State.CLOSING, State.CLOSED);
                }
            }, MoreExecutors.directExecutor());
        } else {
            switch (AnonymousClass11.$SwitchMap$com$google$common$util$concurrent$ClosingFuture$State[this.state.get().ordinal()]) {
                case 1:
                    throw new IllegalStateException("Cannot call finishToFuture() after deriving another step");
                case 2:
                    throw new IllegalStateException("Cannot call finishToFuture() after calling finishToValueAndCloser()");
                case 3:
                case 4:
                case 5:
                    throw new IllegalStateException("Cannot call finishToFuture() twice");
                case 6:
                    throw new AssertionError();
            }
        }
        return this.future;
    }

    /* access modifiers changed from: package-private */
    /* renamed from: com.google.common.util.concurrent.ClosingFuture$11  reason: invalid class name */
    public static /* synthetic */ class AnonymousClass11 {
        static final /* synthetic */ int[] $SwitchMap$com$google$common$util$concurrent$ClosingFuture$State;

        /* JADX WARNING: Can't wrap try/catch for region: R(14:0|1|2|3|4|5|6|7|8|9|10|11|12|14) */
        /* JADX WARNING: Failed to process nested try/catch */
        /* JADX WARNING: Missing exception handler attribute for start block: B:11:0x003e */
        /* JADX WARNING: Missing exception handler attribute for start block: B:3:0x0012 */
        /* JADX WARNING: Missing exception handler attribute for start block: B:5:0x001d */
        /* JADX WARNING: Missing exception handler attribute for start block: B:7:0x0028 */
        /* JADX WARNING: Missing exception handler attribute for start block: B:9:0x0033 */
        static {
            int[] iArr = new int[State.values().length];
            $SwitchMap$com$google$common$util$concurrent$ClosingFuture$State = iArr;
            iArr[State.SUBSUMED.ordinal()] = 1;
            $SwitchMap$com$google$common$util$concurrent$ClosingFuture$State[State.WILL_CREATE_VALUE_AND_CLOSER.ordinal()] = 2;
            $SwitchMap$com$google$common$util$concurrent$ClosingFuture$State[State.WILL_CLOSE.ordinal()] = 3;
            $SwitchMap$com$google$common$util$concurrent$ClosingFuture$State[State.CLOSING.ordinal()] = 4;
            $SwitchMap$com$google$common$util$concurrent$ClosingFuture$State[State.CLOSED.ordinal()] = 5;
            try {
                $SwitchMap$com$google$common$util$concurrent$ClosingFuture$State[State.OPEN.ordinal()] = 6;
            } catch (NoSuchFieldError unused) {
            }
        }
    }

    public void finishToValueAndCloser(final ValueAndCloserConsumer<? super V> valueAndCloserConsumer, Executor executor) {
        Preconditions.checkNotNull(valueAndCloserConsumer);
        if (!compareAndUpdateState(State.OPEN, State.WILL_CREATE_VALUE_AND_CLOSER)) {
            int i = AnonymousClass11.$SwitchMap$com$google$common$util$concurrent$ClosingFuture$State[this.state.get().ordinal()];
            if (i == 1) {
                throw new IllegalStateException("Cannot call finishToValueAndCloser() after deriving another step");
            } else if (i == 2) {
                throw new IllegalStateException("Cannot call finishToValueAndCloser() twice");
            } else if (i == 3 || i == 4 || i == 5) {
                throw new IllegalStateException("Cannot call finishToValueAndCloser() after calling finishToFuture()");
            } else {
                throw new AssertionError(this.state);
            }
        } else {
            this.future.addListener(new Runnable() {
                /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass9 */

                public void run() {
                    ClosingFuture.provideValueAndCloser(valueAndCloserConsumer, ClosingFuture.this);
                }
            }, executor);
        }
    }

    /* access modifiers changed from: private */
    public static <C, V extends C> void provideValueAndCloser(ValueAndCloserConsumer<C> valueAndCloserConsumer, ClosingFuture<V> closingFuture) {
        valueAndCloserConsumer.accept(new ValueAndCloser<>(closingFuture));
    }

    public boolean cancel(boolean z) {
        logger.log(Level.FINER, "cancelling {0}", this);
        boolean cancel = this.future.cancel(z);
        if (cancel) {
            close();
        }
        return cancel;
    }

    /* access modifiers changed from: private */
    /* access modifiers changed from: public */
    private void close() {
        logger.log(Level.FINER, "closing {0}", this);
        this.closeables.close();
    }

    private <U> ClosingFuture<U> derive(FluentFuture<U> fluentFuture) {
        ClosingFuture<U> closingFuture = new ClosingFuture<>(fluentFuture);
        becomeSubsumedInto(closingFuture.closeables);
        return closingFuture;
    }

    /* access modifiers changed from: private */
    /* access modifiers changed from: public */
    private void becomeSubsumedInto(CloseableList closeableList) {
        checkAndUpdateState(State.OPEN, State.SUBSUMED);
        closeableList.add(this.closeables, MoreExecutors.directExecutor());
    }

    public static final class Peeker {
        private volatile boolean beingCalled;
        private final ImmutableList<ClosingFuture<?>> futures;

        private Peeker(ImmutableList<ClosingFuture<?>> immutableList) {
            this.futures = (ImmutableList) Preconditions.checkNotNull(immutableList);
        }

        @NullableDecl
        public final <D> D getDone(ClosingFuture<D> closingFuture) throws ExecutionException {
            Preconditions.checkState(this.beingCalled);
            Preconditions.checkArgument(this.futures.contains(closingFuture));
            return (D) Futures.getDone(((ClosingFuture) closingFuture).future);
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        @NullableDecl
        private <V> V call(Combiner.CombiningCallable<V> combiningCallable, CloseableList closeableList) throws Exception {
            this.beingCalled = true;
            CloseableList closeableList2 = new CloseableList();
            try {
                return combiningCallable.call(closeableList2.closer, this);
            } finally {
                closeableList.add(closeableList2, MoreExecutors.directExecutor());
                this.beingCalled = false;
            }
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private <V> FluentFuture<V> callAsync(Combiner.AsyncCombiningCallable<V> asyncCombiningCallable, CloseableList closeableList) throws Exception {
            this.beingCalled = true;
            CloseableList closeableList2 = new CloseableList();
            try {
                ClosingFuture<V> call = asyncCombiningCallable.call(closeableList2.closer, this);
                call.becomeSubsumedInto(closeableList);
                return ((ClosingFuture) call).future;
            } finally {
                closeableList.add(closeableList2, MoreExecutors.directExecutor());
                this.beingCalled = false;
            }
        }
    }

    @DoNotMock("Use ClosingFuture.whenAllSucceed() or .whenAllComplete() instead.")
    public static class Combiner {
        private static final Function<ClosingFuture<?>, FluentFuture<?>> INNER_FUTURE = new Function<ClosingFuture<?>, FluentFuture<?>>() {
            /* class com.google.common.util.concurrent.ClosingFuture.Combiner.AnonymousClass3 */

            public FluentFuture<?> apply(ClosingFuture<?> closingFuture) {
                return ((ClosingFuture) closingFuture).future;
            }
        };
        private final boolean allMustSucceed;
        private final CloseableList closeables;
        protected final ImmutableList<ClosingFuture<?>> inputs;

        public interface AsyncCombiningCallable<V> {
            ClosingFuture<V> call(DeferredCloser deferredCloser, Peeker peeker) throws Exception;
        }

        public interface CombiningCallable<V> {
            @NullableDecl
            V call(DeferredCloser deferredCloser, Peeker peeker) throws Exception;
        }

        private Combiner(boolean z, Iterable<? extends ClosingFuture<?>> iterable) {
            this.closeables = new CloseableList();
            this.allMustSucceed = z;
            this.inputs = ImmutableList.copyOf(iterable);
            Iterator<? extends ClosingFuture<?>> it = iterable.iterator();
            while (it.hasNext()) {
                ((ClosingFuture) it.next()).becomeSubsumedInto(this.closeables);
            }
        }

        public <V> ClosingFuture<V> call(final CombiningCallable<V> combiningCallable, Executor executor) {
            ClosingFuture<V> closingFuture = new ClosingFuture<>(futureCombiner().call(new Callable<V>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner.AnonymousClass1 */

                @Override // java.util.concurrent.Callable
                public V call() throws Exception {
                    return (V) new Peeker(Combiner.this.inputs).call(combiningCallable, Combiner.this.closeables);
                }

                public String toString() {
                    return combiningCallable.toString();
                }
            }, executor));
            ((ClosingFuture) closingFuture).closeables.add(this.closeables, MoreExecutors.directExecutor());
            return closingFuture;
        }

        public <V> ClosingFuture<V> callAsync(final AsyncCombiningCallable<V> asyncCombiningCallable, Executor executor) {
            ClosingFuture<V> closingFuture = new ClosingFuture<>(futureCombiner().callAsync(new AsyncCallable<V>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner.AnonymousClass2 */

                @Override // com.google.common.util.concurrent.AsyncCallable
                public ListenableFuture<V> call() throws Exception {
                    return new Peeker(Combiner.this.inputs).callAsync(asyncCombiningCallable, Combiner.this.closeables);
                }

                public String toString() {
                    return asyncCombiningCallable.toString();
                }
            }, executor));
            ((ClosingFuture) closingFuture).closeables.add(this.closeables, MoreExecutors.directExecutor());
            return closingFuture;
        }

        private Futures.FutureCombiner<Object> futureCombiner() {
            if (this.allMustSucceed) {
                return Futures.whenAllSucceed(inputFutures());
            }
            return Futures.whenAllComplete(inputFutures());
        }

        private ImmutableList<FluentFuture<?>> inputFutures() {
            return FluentIterable.from(this.inputs).transform(INNER_FUTURE).toList();
        }
    }

    public static final class Combiner2<V1, V2> extends Combiner {
        private final ClosingFuture<V1> future1;
        private final ClosingFuture<V2> future2;

        public interface AsyncClosingFunction2<V1, V2, U> {
            ClosingFuture<U> apply(DeferredCloser deferredCloser, @NullableDecl V1 v1, @NullableDecl V2 v2) throws Exception;
        }

        public interface ClosingFunction2<V1, V2, U> {
            @NullableDecl
            U apply(DeferredCloser deferredCloser, @NullableDecl V1 v1, @NullableDecl V2 v2) throws Exception;
        }

        private Combiner2(ClosingFuture<V1> closingFuture, ClosingFuture<V2> closingFuture2) {
            super(true, ImmutableList.of(closingFuture, closingFuture2));
            this.future1 = closingFuture;
            this.future2 = closingFuture2;
        }

        /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<V>, com.google.common.util.concurrent.ClosingFuture<U> */
        public <U> ClosingFuture<U> call(final ClosingFunction2<V1, V2, U> closingFunction2, Executor executor) {
            return (ClosingFuture<V>) call(new Combiner.CombiningCallable<U>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner2.AnonymousClass1 */

                /* JADX DEBUG: Multi-variable search result rejected for r0v0, resolved type: com.google.common.util.concurrent.ClosingFuture$Combiner2$ClosingFunction2 */
                /* JADX WARN: Multi-variable type inference failed */
                @Override // com.google.common.util.concurrent.ClosingFuture.Combiner.CombiningCallable
                @NullableDecl
                public U call(DeferredCloser deferredCloser, Peeker peeker) throws Exception {
                    return (U) closingFunction2.apply(deferredCloser, peeker.getDone(Combiner2.this.future1), peeker.getDone(Combiner2.this.future2));
                }

                public String toString() {
                    return closingFunction2.toString();
                }
            }, executor);
        }

        /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<V>, com.google.common.util.concurrent.ClosingFuture<U> */
        public <U> ClosingFuture<U> callAsync(final AsyncClosingFunction2<V1, V2, U> asyncClosingFunction2, Executor executor) {
            return (ClosingFuture<V>) callAsync(new Combiner.AsyncCombiningCallable<U>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner2.AnonymousClass2 */

                /* JADX DEBUG: Multi-variable search result rejected for r0v0, resolved type: com.google.common.util.concurrent.ClosingFuture$Combiner2$AsyncClosingFunction2 */
                /* JADX WARN: Multi-variable type inference failed */
                @Override // com.google.common.util.concurrent.ClosingFuture.Combiner.AsyncCombiningCallable
                public ClosingFuture<U> call(DeferredCloser deferredCloser, Peeker peeker) throws Exception {
                    return asyncClosingFunction2.apply(deferredCloser, peeker.getDone(Combiner2.this.future1), peeker.getDone(Combiner2.this.future2));
                }

                public String toString() {
                    return asyncClosingFunction2.toString();
                }
            }, executor);
        }
    }

    public static final class Combiner3<V1, V2, V3> extends Combiner {
        private final ClosingFuture<V1> future1;
        private final ClosingFuture<V2> future2;
        private final ClosingFuture<V3> future3;

        public interface AsyncClosingFunction3<V1, V2, V3, U> {
            ClosingFuture<U> apply(DeferredCloser deferredCloser, @NullableDecl V1 v1, @NullableDecl V2 v2, @NullableDecl V3 v3) throws Exception;
        }

        public interface ClosingFunction3<V1, V2, V3, U> {
            @NullableDecl
            U apply(DeferredCloser deferredCloser, @NullableDecl V1 v1, @NullableDecl V2 v2, @NullableDecl V3 v3) throws Exception;
        }

        private Combiner3(ClosingFuture<V1> closingFuture, ClosingFuture<V2> closingFuture2, ClosingFuture<V3> closingFuture3) {
            super(true, ImmutableList.of(closingFuture, closingFuture2, closingFuture3));
            this.future1 = closingFuture;
            this.future2 = closingFuture2;
            this.future3 = closingFuture3;
        }

        /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<V>, com.google.common.util.concurrent.ClosingFuture<U> */
        public <U> ClosingFuture<U> call(final ClosingFunction3<V1, V2, V3, U> closingFunction3, Executor executor) {
            return (ClosingFuture<V>) call(new Combiner.CombiningCallable<U>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner3.AnonymousClass1 */

                /* JADX DEBUG: Multi-variable search result rejected for r0v0, resolved type: com.google.common.util.concurrent.ClosingFuture$Combiner3$ClosingFunction3 */
                /* JADX WARN: Multi-variable type inference failed */
                @Override // com.google.common.util.concurrent.ClosingFuture.Combiner.CombiningCallable
                @NullableDecl
                public U call(DeferredCloser deferredCloser, Peeker peeker) throws Exception {
                    return (U) closingFunction3.apply(deferredCloser, peeker.getDone(Combiner3.this.future1), peeker.getDone(Combiner3.this.future2), peeker.getDone(Combiner3.this.future3));
                }

                public String toString() {
                    return closingFunction3.toString();
                }
            }, executor);
        }

        /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<V>, com.google.common.util.concurrent.ClosingFuture<U> */
        public <U> ClosingFuture<U> callAsync(final AsyncClosingFunction3<V1, V2, V3, U> asyncClosingFunction3, Executor executor) {
            return (ClosingFuture<V>) callAsync(new Combiner.AsyncCombiningCallable<U>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner3.AnonymousClass2 */

                /* JADX DEBUG: Multi-variable search result rejected for r0v0, resolved type: com.google.common.util.concurrent.ClosingFuture$Combiner3$AsyncClosingFunction3 */
                /* JADX WARN: Multi-variable type inference failed */
                @Override // com.google.common.util.concurrent.ClosingFuture.Combiner.AsyncCombiningCallable
                public ClosingFuture<U> call(DeferredCloser deferredCloser, Peeker peeker) throws Exception {
                    return asyncClosingFunction3.apply(deferredCloser, peeker.getDone(Combiner3.this.future1), peeker.getDone(Combiner3.this.future2), peeker.getDone(Combiner3.this.future3));
                }

                public String toString() {
                    return asyncClosingFunction3.toString();
                }
            }, executor);
        }
    }

    public static final class Combiner4<V1, V2, V3, V4> extends Combiner {
        private final ClosingFuture<V1> future1;
        private final ClosingFuture<V2> future2;
        private final ClosingFuture<V3> future3;
        private final ClosingFuture<V4> future4;

        public interface AsyncClosingFunction4<V1, V2, V3, V4, U> {
            ClosingFuture<U> apply(DeferredCloser deferredCloser, @NullableDecl V1 v1, @NullableDecl V2 v2, @NullableDecl V3 v3, @NullableDecl V4 v4) throws Exception;
        }

        public interface ClosingFunction4<V1, V2, V3, V4, U> {
            @NullableDecl
            U apply(DeferredCloser deferredCloser, @NullableDecl V1 v1, @NullableDecl V2 v2, @NullableDecl V3 v3, @NullableDecl V4 v4) throws Exception;
        }

        private Combiner4(ClosingFuture<V1> closingFuture, ClosingFuture<V2> closingFuture2, ClosingFuture<V3> closingFuture3, ClosingFuture<V4> closingFuture4) {
            super(true, ImmutableList.of(closingFuture, closingFuture2, closingFuture3, closingFuture4));
            this.future1 = closingFuture;
            this.future2 = closingFuture2;
            this.future3 = closingFuture3;
            this.future4 = closingFuture4;
        }

        /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<V>, com.google.common.util.concurrent.ClosingFuture<U> */
        public <U> ClosingFuture<U> call(final ClosingFunction4<V1, V2, V3, V4, U> closingFunction4, Executor executor) {
            return (ClosingFuture<V>) call(new Combiner.CombiningCallable<U>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner4.AnonymousClass1 */

                /* JADX DEBUG: Multi-variable search result rejected for r0v0, resolved type: com.google.common.util.concurrent.ClosingFuture$Combiner4$ClosingFunction4 */
                /* JADX WARN: Multi-variable type inference failed */
                @Override // com.google.common.util.concurrent.ClosingFuture.Combiner.CombiningCallable
                @NullableDecl
                public U call(DeferredCloser deferredCloser, Peeker peeker) throws Exception {
                    return (U) closingFunction4.apply(deferredCloser, peeker.getDone(Combiner4.this.future1), peeker.getDone(Combiner4.this.future2), peeker.getDone(Combiner4.this.future3), peeker.getDone(Combiner4.this.future4));
                }

                public String toString() {
                    return closingFunction4.toString();
                }
            }, executor);
        }

        /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<V>, com.google.common.util.concurrent.ClosingFuture<U> */
        public <U> ClosingFuture<U> callAsync(final AsyncClosingFunction4<V1, V2, V3, V4, U> asyncClosingFunction4, Executor executor) {
            return (ClosingFuture<V>) callAsync(new Combiner.AsyncCombiningCallable<U>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner4.AnonymousClass2 */

                /* JADX DEBUG: Multi-variable search result rejected for r0v0, resolved type: com.google.common.util.concurrent.ClosingFuture$Combiner4$AsyncClosingFunction4 */
                /* JADX WARN: Multi-variable type inference failed */
                @Override // com.google.common.util.concurrent.ClosingFuture.Combiner.AsyncCombiningCallable
                public ClosingFuture<U> call(DeferredCloser deferredCloser, Peeker peeker) throws Exception {
                    return asyncClosingFunction4.apply(deferredCloser, peeker.getDone(Combiner4.this.future1), peeker.getDone(Combiner4.this.future2), peeker.getDone(Combiner4.this.future3), peeker.getDone(Combiner4.this.future4));
                }

                public String toString() {
                    return asyncClosingFunction4.toString();
                }
            }, executor);
        }
    }

    public static final class Combiner5<V1, V2, V3, V4, V5> extends Combiner {
        private final ClosingFuture<V1> future1;
        private final ClosingFuture<V2> future2;
        private final ClosingFuture<V3> future3;
        private final ClosingFuture<V4> future4;
        private final ClosingFuture<V5> future5;

        public interface AsyncClosingFunction5<V1, V2, V3, V4, V5, U> {
            ClosingFuture<U> apply(DeferredCloser deferredCloser, @NullableDecl V1 v1, @NullableDecl V2 v2, @NullableDecl V3 v3, @NullableDecl V4 v4, @NullableDecl V5 v5) throws Exception;
        }

        public interface ClosingFunction5<V1, V2, V3, V4, V5, U> {
            @NullableDecl
            U apply(DeferredCloser deferredCloser, @NullableDecl V1 v1, @NullableDecl V2 v2, @NullableDecl V3 v3, @NullableDecl V4 v4, @NullableDecl V5 v5) throws Exception;
        }

        private Combiner5(ClosingFuture<V1> closingFuture, ClosingFuture<V2> closingFuture2, ClosingFuture<V3> closingFuture3, ClosingFuture<V4> closingFuture4, ClosingFuture<V5> closingFuture5) {
            super(true, ImmutableList.of(closingFuture, closingFuture2, closingFuture3, closingFuture4, closingFuture5));
            this.future1 = closingFuture;
            this.future2 = closingFuture2;
            this.future3 = closingFuture3;
            this.future4 = closingFuture4;
            this.future5 = closingFuture5;
        }

        /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<V>, com.google.common.util.concurrent.ClosingFuture<U> */
        public <U> ClosingFuture<U> call(final ClosingFunction5<V1, V2, V3, V4, V5, U> closingFunction5, Executor executor) {
            return (ClosingFuture<V>) call(new Combiner.CombiningCallable<U>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner5.AnonymousClass1 */

                /* JADX DEBUG: Multi-variable search result rejected for r0v0, resolved type: com.google.common.util.concurrent.ClosingFuture$Combiner5$ClosingFunction5 */
                /* JADX WARN: Multi-variable type inference failed */
                @Override // com.google.common.util.concurrent.ClosingFuture.Combiner.CombiningCallable
                @NullableDecl
                public U call(DeferredCloser deferredCloser, Peeker peeker) throws Exception {
                    return (U) closingFunction5.apply(deferredCloser, peeker.getDone(Combiner5.this.future1), peeker.getDone(Combiner5.this.future2), peeker.getDone(Combiner5.this.future3), peeker.getDone(Combiner5.this.future4), peeker.getDone(Combiner5.this.future5));
                }

                public String toString() {
                    return closingFunction5.toString();
                }
            }, executor);
        }

        /* JADX DEBUG: Type inference failed for r2v1. Raw type applied. Possible types: com.google.common.util.concurrent.ClosingFuture<V>, com.google.common.util.concurrent.ClosingFuture<U> */
        public <U> ClosingFuture<U> callAsync(final AsyncClosingFunction5<V1, V2, V3, V4, V5, U> asyncClosingFunction5, Executor executor) {
            return (ClosingFuture<V>) callAsync(new Combiner.AsyncCombiningCallable<U>() {
                /* class com.google.common.util.concurrent.ClosingFuture.Combiner5.AnonymousClass2 */

                /* JADX DEBUG: Multi-variable search result rejected for r0v0, resolved type: com.google.common.util.concurrent.ClosingFuture$Combiner5$AsyncClosingFunction5 */
                /* JADX WARN: Multi-variable type inference failed */
                @Override // com.google.common.util.concurrent.ClosingFuture.Combiner.AsyncCombiningCallable
                public ClosingFuture<U> call(DeferredCloser deferredCloser, Peeker peeker) throws Exception {
                    return asyncClosingFunction5.apply(deferredCloser, peeker.getDone(Combiner5.this.future1), peeker.getDone(Combiner5.this.future2), peeker.getDone(Combiner5.this.future3), peeker.getDone(Combiner5.this.future4), peeker.getDone(Combiner5.this.future5));
                }

                public String toString() {
                    return asyncClosingFunction5.toString();
                }
            }, executor);
        }
    }

    public String toString() {
        return MoreObjects.toStringHelper(this).add("state", this.state.get()).addValue(this.future).toString();
    }

    /* access modifiers changed from: protected */
    public void finalize() {
        if (this.state.get().equals(State.OPEN)) {
            logger.log(Level.SEVERE, "Uh oh! An open ClosingFuture has leaked and will close: {0}", this);
            finishToFuture();
        }
    }

    /* access modifiers changed from: private */
    public static void closeQuietly(final Closeable closeable, Executor executor) {
        if (closeable != null) {
            try {
                executor.execute(new Runnable() {
                    /* class com.google.common.util.concurrent.ClosingFuture.AnonymousClass10 */

                    public void run() {
                        try {
                            closeable.close();
                        } catch (IOException | RuntimeException e) {
                            ClosingFuture.logger.log(Level.WARNING, "thrown by close()", e);
                        }
                    }
                });
            } catch (RejectedExecutionException e) {
                Logger logger2 = logger;
                if (logger2.isLoggable(Level.WARNING)) {
                    logger2.log(Level.WARNING, String.format("while submitting close to %s; will close inline", executor), (Throwable) e);
                }
                closeQuietly(closeable, MoreExecutors.directExecutor());
            }
        }
    }

    /* access modifiers changed from: private */
    /* access modifiers changed from: public */
    private void checkAndUpdateState(State state2, State state3) {
        Preconditions.checkState(compareAndUpdateState(state2, state3), "Expected state to be %s, but it was %s", state2, state3);
    }

    private boolean compareAndUpdateState(State state2, State state3) {
        return this.state.compareAndSet(state2, state3);
    }

    /* access modifiers changed from: private */
    public static final class CloseableList extends IdentityHashMap<Closeable, Executor> implements Closeable {
        private volatile boolean closed;
        private final DeferredCloser closer;
        private volatile CountDownLatch whenClosed;

        private CloseableList() {
            this.closer = new DeferredCloser(this);
        }

        /* access modifiers changed from: package-private */
        public <V, U> ListenableFuture<U> applyClosingFunction(ClosingFunction<? super V, U> closingFunction, V v) throws Exception {
            CloseableList closeableList = new CloseableList();
            try {
                return Futures.immediateFuture(closingFunction.apply(closeableList.closer, v));
            } finally {
                add(closeableList, MoreExecutors.directExecutor());
            }
        }

        /* access modifiers changed from: package-private */
        public <V, U> FluentFuture<U> applyAsyncClosingFunction(AsyncClosingFunction<V, U> asyncClosingFunction, V v) throws Exception {
            CloseableList closeableList = new CloseableList();
            try {
                ClosingFuture<U> apply = asyncClosingFunction.apply(closeableList.closer, v);
                apply.becomeSubsumedInto(closeableList);
                return ((ClosingFuture) apply).future;
            } finally {
                add(closeableList, MoreExecutors.directExecutor());
            }
        }

        /* JADX WARNING: Code restructure failed: missing block: B:11:0x0010, code lost:
            r0 = entrySet().iterator();
         */
        /* JADX WARNING: Code restructure failed: missing block: B:13:0x001c, code lost:
            if (r0.hasNext() == false) goto L_0x0034;
         */
        /* JADX WARNING: Code restructure failed: missing block: B:14:0x001e, code lost:
            r1 = (java.util.Map.Entry) r0.next();
            com.google.common.util.concurrent.ClosingFuture.closeQuietly((java.io.Closeable) r1.getKey(), (java.util.concurrent.Executor) r1.getValue());
         */
        /* JADX WARNING: Code restructure failed: missing block: B:15:0x0034, code lost:
            clear();
         */
        /* JADX WARNING: Code restructure failed: missing block: B:16:0x0039, code lost:
            if (r3.whenClosed == null) goto L_?;
         */
        /* JADX WARNING: Code restructure failed: missing block: B:17:0x003b, code lost:
            r3.whenClosed.countDown();
         */
        /* JADX WARNING: Code restructure failed: missing block: B:22:?, code lost:
            return;
         */
        /* JADX WARNING: Code restructure failed: missing block: B:23:?, code lost:
            return;
         */
        @Override // java.io.Closeable, java.lang.AutoCloseable
        public void close() {
            if (!this.closed) {
                synchronized (this) {
                    if (!this.closed) {
                        this.closed = true;
                    }
                }
            }
        }

        /* access modifiers changed from: package-private */
        public void add(@NullableDecl Closeable closeable, Executor executor) {
            Preconditions.checkNotNull(executor);
            if (closeable != null) {
                synchronized (this) {
                    if (!this.closed) {
                        put(closeable, executor);
                    } else {
                        ClosingFuture.closeQuietly(closeable, executor);
                    }
                }
            }
        }

        /* access modifiers changed from: package-private */
        public CountDownLatch whenClosedCountDown() {
            boolean z = false;
            if (this.closed) {
                return new CountDownLatch(0);
            }
            synchronized (this) {
                if (this.closed) {
                    return new CountDownLatch(0);
                }
                if (this.whenClosed == null) {
                    z = true;
                }
                Preconditions.checkState(z);
                CountDownLatch countDownLatch = new CountDownLatch(1);
                this.whenClosed = countDownLatch;
                return countDownLatch;
            }
        }
    }

    /* access modifiers changed from: package-private */
    public CountDownLatch whenClosedCountDown() {
        return this.closeables.whenClosedCountDown();
    }
}
