package net.lingala.zip4j.tasks;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.lingala.zip4j.exception.ZipException;
import net.lingala.zip4j.headers.HeaderUtil;
import net.lingala.zip4j.headers.HeaderWriter;
import net.lingala.zip4j.io.outputstream.SplitOutputStream;
import net.lingala.zip4j.model.FileHeader;
import net.lingala.zip4j.model.Zip4jConfig;
import net.lingala.zip4j.model.ZipModel;
import net.lingala.zip4j.model.enums.RandomAccessFileMode;
import net.lingala.zip4j.progress.ProgressMonitor;
import net.lingala.zip4j.tasks.AsyncZipTask;
import net.lingala.zip4j.util.RawIO;
import net.lingala.zip4j.util.Zip4jUtil;

public class RenameFilesTask extends AbstractModifyFileTask<RenameFilesTaskParameters> {
    private final HeaderWriter headerWriter;
    private final RawIO rawIO;
    private final ZipModel zipModel;

    public static class RenameFilesTaskParameters extends AbstractZipTaskParameters {
        private final Map<String, String> fileNamesMap;

        public RenameFilesTaskParameters(Map<String, String> map, Zip4jConfig zip4jConfig) {
            super(zip4jConfig);
            this.fileNamesMap = map;
        }
    }

    public RenameFilesTask(ZipModel zipModel2, HeaderWriter headerWriter2, RawIO rawIO2, AsyncZipTask.AsyncTaskParameters asyncTaskParameters) {
        super(asyncTaskParameters);
        this.zipModel = zipModel2;
        this.headerWriter = headerWriter2;
        this.rawIO = rawIO2;
    }

    private long copyEntryAndChangeFileName(byte[] bArr, FileHeader fileHeader, long j, long j2, RandomAccessFile randomAccessFile, OutputStream outputStream, ProgressMonitor progressMonitor, int i) throws IOException {
        this.rawIO.writeShortLittleEndian(outputStream, bArr.length);
        long copyFile = j + copyFile(randomAccessFile, outputStream, j, 26, progressMonitor, i) + 2;
        outputStream.write(bArr);
        long copyFile2 = copyFile + copyFile(randomAccessFile, outputStream, copyFile, 2, progressMonitor, i) + ((long) fileHeader.getFileNameLength());
        return copyFile2 + copyFile(randomAccessFile, outputStream, copyFile2, j2 - (copyFile2 - j), progressMonitor, i);
    }

    private Map<String, String> filterNonExistingEntriesAndAddSeparatorIfNeeded(Map<String, String> map) throws ZipException {
        FileHeader fileHeader;
        HashMap hashMap = new HashMap();
        for (Map.Entry<String, String> entry : map.entrySet()) {
            if (Zip4jUtil.isStringNotNullAndNotEmpty(entry.getKey()) && (fileHeader = HeaderUtil.getFileHeader(this.zipModel, entry.getKey())) != null) {
                if (!fileHeader.isDirectory() || entry.getValue().endsWith("/")) {
                    hashMap.put(entry.getKey(), entry.getValue());
                } else {
                    String key = entry.getKey();
                    hashMap.put(key, entry.getValue() + "/");
                }
            }
        }
        return hashMap;
    }

    private Map.Entry<String, String> getCorrespondingEntryFromMap(FileHeader fileHeader, Map<String, String> map) {
        for (Map.Entry<String, String> entry : map.entrySet()) {
            if (fileHeader.getFileName().startsWith(entry.getKey())) {
                return entry;
            }
        }
        return null;
    }

    private String getNewFileName(String str, String str2, String str3) throws ZipException {
        if (str3.equals(str2)) {
            return str;
        }
        if (str3.startsWith(str2)) {
            String substring = str3.substring(str2.length());
            return str + substring;
        }
        throw new ZipException("old file name was neither an exact match nor a partial match");
    }

    private void updateHeadersInZipModel(List<FileHeader> list, FileHeader fileHeader, String str, byte[] bArr, int i) throws ZipException {
        FileHeader fileHeader2 = HeaderUtil.getFileHeader(this.zipModel, fileHeader.getFileName());
        if (fileHeader2 != null) {
            fileHeader2.setFileName(str);
            fileHeader2.setFileNameLength(bArr.length);
            long j = (long) i;
            updateOffsetsForAllSubsequentFileHeaders(list, this.zipModel, fileHeader2, j);
            this.zipModel.getEndOfCentralDirectoryRecord().setOffsetOfStartOfCentralDirectory(this.zipModel.getEndOfCentralDirectoryRecord().getOffsetOfStartOfCentralDirectory() + j);
            if (this.zipModel.isZip64Format()) {
                this.zipModel.getZip64EndOfCentralDirectoryRecord().setOffsetStartCentralDirectoryWRTStartDiskNumber(this.zipModel.getZip64EndOfCentralDirectoryRecord().getOffsetStartCentralDirectoryWRTStartDiskNumber() + j);
                this.zipModel.getZip64EndOfCentralDirectoryLocator().setOffsetZip64EndOfCentralDirectoryRecord(this.zipModel.getZip64EndOfCentralDirectoryLocator().getOffsetZip64EndOfCentralDirectoryRecord() + j);
                return;
            }
            return;
        }
        throw new ZipException("could not find any header with name: " + fileHeader.getFileName());
    }

    /* access modifiers changed from: protected */
    @Override // net.lingala.zip4j.tasks.AsyncZipTask
    public ProgressMonitor.Task getTask() {
        return ProgressMonitor.Task.RENAME_FILE;
    }

    /* access modifiers changed from: protected */
    public long calculateTotalWork(RenameFilesTaskParameters renameFilesTaskParameters) {
        return this.zipModel.getZipFile().length();
    }

    /* access modifiers changed from: protected */
    public void executeTask(RenameFilesTaskParameters renameFilesTaskParameters, ProgressMonitor progressMonitor) throws IOException {
        boolean z;
        Throwable th;
        RandomAccessFile randomAccessFile;
        Throwable th2;
        Throwable th3;
        SplitOutputStream splitOutputStream;
        Throwable th4;
        Throwable th5;
        List<FileHeader> list;
        Map<String, String> map;
        Charset charset;
        Map<String, String> filterNonExistingEntriesAndAddSeparatorIfNeeded = filterNonExistingEntriesAndAddSeparatorIfNeeded(renameFilesTaskParameters.fileNamesMap);
        if (filterNonExistingEntriesAndAddSeparatorIfNeeded.size() != 0) {
            File temporaryFile = getTemporaryFile(this.zipModel.getZipFile().getPath());
            try {
                RandomAccessFile randomAccessFile2 = new RandomAccessFile(this.zipModel.getZipFile(), RandomAccessFileMode.WRITE.getValue());
                try {
                    SplitOutputStream splitOutputStream2 = new SplitOutputStream(temporaryFile);
                    try {
                        Charset charset2 = renameFilesTaskParameters.zip4jConfig.getCharset();
                        List<FileHeader> cloneAndSortFileHeadersByOffset = cloneAndSortFileHeadersByOffset(this.zipModel.getCentralDirectory().getFileHeaders());
                        long j = 0;
                        for (FileHeader fileHeader : cloneAndSortFileHeadersByOffset) {
                            Map.Entry<String, String> correspondingEntryFromMap = getCorrespondingEntryFromMap(fileHeader, filterNonExistingEntriesAndAddSeparatorIfNeeded);
                            progressMonitor.setFileName(fileHeader.getFileName());
                            long offsetOfNextEntry = getOffsetOfNextEntry(cloneAndSortFileHeadersByOffset, fileHeader, this.zipModel) - splitOutputStream2.getFilePointer();
                            if (correspondingEntryFromMap == null) {
                                try {
                                    list = cloneAndSortFileHeadersByOffset;
                                    charset = charset2;
                                    j += copyFile(randomAccessFile2, splitOutputStream2, j, offsetOfNextEntry, progressMonitor, renameFilesTaskParameters.zip4jConfig.getBufferSize());
                                    randomAccessFile = randomAccessFile2;
                                    map = filterNonExistingEntriesAndAddSeparatorIfNeeded;
                                    splitOutputStream = splitOutputStream2;
                                } catch (Throwable th6) {
                                    th4 = th6;
                                    splitOutputStream = splitOutputStream2;
                                    randomAccessFile = randomAccessFile2;
                                    try {
                                        splitOutputStream.close();
                                    } catch (Throwable th7) {
                                        th3 = th7;
                                        th2 = th3;
                                        try {
                                            randomAccessFile.close();
                                        } catch (Throwable th8) {
                                            th2.addSuppressed(th8);
                                        }
                                        throw th2;
                                    }
                                    throw th4;
                                }
                            } else {
                                list = cloneAndSortFileHeadersByOffset;
                                charset = charset2;
                                String newFileName = getNewFileName(correspondingEntryFromMap.getValue(), correspondingEntryFromMap.getKey(), fileHeader.getFileName());
                                byte[] bytesFromString = HeaderUtil.getBytesFromString(newFileName, charset);
                                int length = bytesFromString.length - fileHeader.getFileNameLength();
                                map = filterNonExistingEntriesAndAddSeparatorIfNeeded;
                                splitOutputStream = splitOutputStream2;
                                randomAccessFile = randomAccessFile2;
                                try {
                                    long copyEntryAndChangeFileName = copyEntryAndChangeFileName(bytesFromString, fileHeader, j, offsetOfNextEntry, randomAccessFile2, splitOutputStream2, progressMonitor, renameFilesTaskParameters.zip4jConfig.getBufferSize());
                                    updateHeadersInZipModel(list, fileHeader, newFileName, bytesFromString, length);
                                    j = copyEntryAndChangeFileName;
                                } catch (Throwable th9) {
                                    th5 = th9;
                                    th4 = th5;
                                    splitOutputStream.close();
                                    throw th4;
                                }
                            }
                            verifyIfTaskIsCancelled();
                            splitOutputStream2 = splitOutputStream;
                            charset2 = charset;
                            filterNonExistingEntriesAndAddSeparatorIfNeeded = map;
                            cloneAndSortFileHeadersByOffset = list;
                            randomAccessFile2 = randomAccessFile;
                        }
                        randomAccessFile = randomAccessFile2;
                        this.headerWriter.finalizeZipFile(this.zipModel, splitOutputStream2, charset2);
                        z = true;
                    } catch (Throwable th10) {
                        th5 = th10;
                        splitOutputStream = splitOutputStream2;
                        randomAccessFile = randomAccessFile2;
                        th4 = th5;
                        splitOutputStream.close();
                        throw th4;
                    }
                    try {
                        splitOutputStream2.close();
                        try {
                            randomAccessFile.close();
                            cleanupFile(true, this.zipModel.getZipFile(), temporaryFile);
                        } catch (Throwable th11) {
                            th = th11;
                            cleanupFile(z, this.zipModel.getZipFile(), temporaryFile);
                            throw th;
                        }
                    } catch (Throwable th12) {
                        th2 = th12;
                        randomAccessFile.close();
                        throw th2;
                    }
                } catch (Throwable th13) {
                    th3 = th13;
                    randomAccessFile = randomAccessFile2;
                    th2 = th3;
                    randomAccessFile.close();
                    throw th2;
                }
            } catch (Throwable th14) {
                th = th14;
                z = false;
                cleanupFile(z, this.zipModel.getZipFile(), temporaryFile);
                throw th;
            }
        }
    }
}
