package master.flame.danmaku.controller;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import master.flame.danmaku.controller.IDrawTask;
import master.flame.danmaku.danmaku.model.AbsDisplayer;
import master.flame.danmaku.danmaku.model.BaseDanmaku;
import master.flame.danmaku.danmaku.model.DanmakuTimer;
import master.flame.danmaku.danmaku.model.ICacheManager;
import master.flame.danmaku.danmaku.model.IDanmakuIterator;
import master.flame.danmaku.danmaku.model.IDanmakus;
import master.flame.danmaku.danmaku.model.IDrawingCache;
import master.flame.danmaku.danmaku.model.android.DanmakuContext;
import master.flame.danmaku.danmaku.model.android.DanmakuFactory;
import master.flame.danmaku.danmaku.model.android.Danmakus;
import master.flame.danmaku.danmaku.model.android.DrawingCache;
import master.flame.danmaku.danmaku.model.android.DrawingCachePoolManager;
import master.flame.danmaku.danmaku.model.objectpool.Pool;
import master.flame.danmaku.danmaku.model.objectpool.Pools;
import master.flame.danmaku.danmaku.renderer.IRenderer;
import master.flame.danmaku.danmaku.util.DanmakuUtils;
import master.flame.danmaku.danmaku.util.SystemClock;
import tv.cjump.jni.NativeBitmapFactory;

public class CacheManagingDrawTask extends DrawTask {
    static final /* synthetic */ boolean $assertionsDisabled = false;
    private static final int MAX_CACHE_SCREEN_SIZE = 3;
    private CacheManager mCacheManager;
    private DanmakuTimer mCacheTimer;
    private final Object mDrawingNotify = new Object();
    private int mMaxCacheSize = 2;
    private int mRemaininCacheCount;

    public CacheManagingDrawTask(DanmakuTimer danmakuTimer, DanmakuContext danmakuContext, IDrawTask.TaskListener taskListener, int i) {
        super(danmakuTimer, danmakuContext, taskListener);
        NativeBitmapFactory.loadLibs();
        this.mMaxCacheSize = i;
        if (NativeBitmapFactory.isInNativeAlloc()) {
            this.mMaxCacheSize = i * 2;
        }
        this.mCacheManager = new CacheManager(i, 3);
        this.mRenderer.setCacheManager(this.mCacheManager);
    }

    /* access modifiers changed from: protected */
    @Override // master.flame.danmaku.controller.DrawTask
    public void initTimer(DanmakuTimer danmakuTimer) {
        this.mTimer = danmakuTimer;
        DanmakuTimer danmakuTimer2 = new DanmakuTimer();
        this.mCacheTimer = danmakuTimer2;
        danmakuTimer2.update(danmakuTimer.currMillisecond);
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void addDanmaku(BaseDanmaku baseDanmaku) {
        super.addDanmaku(baseDanmaku);
        CacheManager cacheManager = this.mCacheManager;
        if (cacheManager != null) {
            cacheManager.addDanmaku(baseDanmaku);
        }
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void invalidateDanmaku(BaseDanmaku baseDanmaku, boolean z) {
        CacheManager cacheManager = this.mCacheManager;
        if (cacheManager == null) {
            super.invalidateDanmaku(baseDanmaku, z);
        } else {
            cacheManager.invalidateDanmaku(baseDanmaku, z);
        }
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void removeAllDanmakus(boolean z) {
        super.removeAllDanmakus(z);
        CacheManager cacheManager = this.mCacheManager;
        if (cacheManager != null) {
            cacheManager.requestClearAll();
        }
    }

    /* access modifiers changed from: protected */
    @Override // master.flame.danmaku.controller.DrawTask
    public void onDanmakuRemoved(BaseDanmaku baseDanmaku) {
        super.onDanmakuRemoved(baseDanmaku);
        CacheManager cacheManager = this.mCacheManager;
        if (cacheManager != null) {
            int i = this.mRemaininCacheCount + 1;
            this.mRemaininCacheCount = i;
            if (i > 5) {
                cacheManager.requestClearTimeout();
                this.mRemaininCacheCount = 0;
                return;
            }
            return;
        }
        IDrawingCache<?> drawingCache = baseDanmaku.getDrawingCache();
        if (drawingCache != null) {
            if (drawingCache.hasReferences()) {
                drawingCache.decreaseReference();
            } else {
                drawingCache.destroy();
            }
            baseDanmaku.cache = null;
        }
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public IRenderer.RenderingState draw(AbsDisplayer absDisplayer) {
        IRenderer.RenderingState draw = super.draw(absDisplayer);
        synchronized (this.mDrawingNotify) {
            this.mDrawingNotify.notify();
        }
        if (!(draw == null || this.mCacheManager == null || draw.totalDanmakuCount - draw.lastTotalDanmakuCount >= -20)) {
            this.mCacheManager.requestClearTimeout();
            this.mCacheManager.requestBuild(-this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION);
        }
        return draw;
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void seek(long j) {
        super.seek(j);
        if (this.mCacheManager == null) {
            start();
        }
        this.mCacheManager.seek(j);
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void start() {
        super.start();
        NativeBitmapFactory.loadLibs();
        CacheManager cacheManager = this.mCacheManager;
        if (cacheManager == null) {
            CacheManager cacheManager2 = new CacheManager(this.mMaxCacheSize, 3);
            this.mCacheManager = cacheManager2;
            cacheManager2.begin();
            this.mRenderer.setCacheManager(this.mCacheManager);
            return;
        }
        cacheManager.resume();
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void quit() {
        super.quit();
        reset();
        this.mRenderer.setCacheManager(null);
        CacheManager cacheManager = this.mCacheManager;
        if (cacheManager != null) {
            cacheManager.end();
            this.mCacheManager = null;
        }
        NativeBitmapFactory.releaseLibs();
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void prepare() {
        loadDanmakus(this.mParser);
        this.mCacheManager.begin();
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void onPlayStateChanged(int i) {
        super.onPlayStateChanged(i);
        CacheManager cacheManager = this.mCacheManager;
        if (cacheManager != null) {
            cacheManager.onPlayStateChanged(i);
        }
    }

    @Override // master.flame.danmaku.controller.IDrawTask, master.flame.danmaku.controller.DrawTask
    public void requestSync(long j, long j2, long j3) {
        super.requestSync(j, j2, j3);
        CacheManager cacheManager = this.mCacheManager;
        if (cacheManager != null) {
            cacheManager.seek(j2);
        }
    }

    public class CacheManager implements ICacheManager {
        public static final byte RESULT_FAILED = 1;
        public static final byte RESULT_FAILED_OVERSIZE = 2;
        public static final byte RESULT_SUCCESS = 0;
        private static final String TAG = "CacheManager";
        Pool<DrawingCache> mCachePool;
        DrawingCachePoolManager mCachePoolManager;
        Danmakus mCaches = new Danmakus();
        private boolean mEndFlag;
        private CacheHandler mHandler;
        private int mMaxSize;
        private int mRealSize;
        private int mScreenSize;
        public HandlerThread mThread;

        public CacheManager(int i, int i2) {
            DrawingCachePoolManager drawingCachePoolManager = new DrawingCachePoolManager();
            this.mCachePoolManager = drawingCachePoolManager;
            this.mCachePool = Pools.finitePool(drawingCachePoolManager, 800);
            this.mScreenSize = 3;
            this.mEndFlag = false;
            this.mRealSize = 0;
            this.mMaxSize = i;
            this.mScreenSize = i2;
        }

        public void seek(long j) {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.requestCancelCaching();
                this.mHandler.removeMessages(3);
                this.mHandler.obtainMessage(5, Long.valueOf(j)).sendToTarget();
            }
        }

        @Override // master.flame.danmaku.danmaku.model.ICacheManager
        public void addDanmaku(BaseDanmaku baseDanmaku) {
            if (this.mHandler == null) {
                return;
            }
            if (!baseDanmaku.isLive) {
                this.mHandler.obtainMessage(2, baseDanmaku).sendToTarget();
            } else if (!baseDanmaku.forceBuildCacheInSameThread) {
                this.mHandler.obtainMessage(18, baseDanmaku).sendToTarget();
            } else if (!baseDanmaku.isTimeOut()) {
                this.mHandler.createCache(baseDanmaku);
            }
        }

        public void invalidateDanmaku(BaseDanmaku baseDanmaku, boolean z) {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.requestCancelCaching();
                this.mHandler.obtainMessage(17, baseDanmaku).sendToTarget();
            }
        }

        public void begin() {
            this.mEndFlag = false;
            if (this.mThread == null) {
                HandlerThread handlerThread = new HandlerThread("DFM Cache-Building Thread");
                this.mThread = handlerThread;
                handlerThread.start();
            }
            if (this.mHandler == null) {
                this.mHandler = new CacheHandler(this.mThread.getLooper());
            }
            this.mHandler.begin();
        }

        public void end() {
            this.mEndFlag = true;
            synchronized (CacheManagingDrawTask.this.mDrawingNotify) {
                CacheManagingDrawTask.this.mDrawingNotify.notifyAll();
            }
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.pause();
                this.mHandler = null;
            }
            HandlerThread handlerThread = this.mThread;
            if (handlerThread != null) {
                try {
                    handlerThread.join();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
                this.mThread.quit();
                this.mThread = null;
            }
        }

        public void resume() {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.resume();
            } else {
                begin();
            }
        }

        public void onPlayStateChanged(int i) {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                boolean z = true;
                if (i != 1) {
                    z = false;
                }
                cacheHandler.onPlayStateChanged(z);
            }
        }

        public float getPoolPercent() {
            int i = this.mMaxSize;
            if (i == 0) {
                return 0.0f;
            }
            return ((float) this.mRealSize) / ((float) i);
        }

        public boolean isPoolFull() {
            return this.mRealSize + 5120 >= this.mMaxSize;
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private void evictAll() {
            Danmakus danmakus = this.mCaches;
            if (danmakus != null) {
                IDanmakuIterator it = danmakus.iterator();
                while (it.hasNext()) {
                    entryRemoved(true, it.next(), null);
                }
                this.mCaches.clear();
            }
            this.mRealSize = 0;
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private void evictAllNotInScreen() {
            evictAllNotInScreen(false);
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private void evictAllNotInScreen(boolean z) {
            Danmakus danmakus = this.mCaches;
            if (danmakus != null) {
                IDanmakuIterator it = danmakus.iterator();
                while (it.hasNext()) {
                    BaseDanmaku next = it.next();
                    IDrawingCache<?> iDrawingCache = next.cache;
                    boolean z2 = iDrawingCache != null && iDrawingCache.hasReferences();
                    if (z && z2) {
                        if (iDrawingCache.get() != null) {
                            this.mRealSize -= iDrawingCache.size();
                            iDrawingCache.destroy();
                        }
                        entryRemoved(true, next, null);
                        it.remove();
                    } else if (next.isOutside()) {
                        entryRemoved(true, next, null);
                        it.remove();
                    }
                }
            }
            this.mRealSize = 0;
        }

        /* access modifiers changed from: protected */
        public void entryRemoved(boolean z, BaseDanmaku baseDanmaku, BaseDanmaku baseDanmaku2) {
            IDrawingCache<?> drawingCache = baseDanmaku.getDrawingCache();
            if (drawingCache != null) {
                long clearCache = clearCache(baseDanmaku);
                if (baseDanmaku.isTimeOut()) {
                    CacheManagingDrawTask.this.mContext.getDisplayer().getCacheStuffer().releaseResource(baseDanmaku);
                }
                if (clearCache > 0) {
                    this.mRealSize = (int) (((long) this.mRealSize) - clearCache);
                    this.mCachePool.release((DrawingCache) drawingCache);
                }
            }
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private long clearCache(BaseDanmaku baseDanmaku) {
            IDrawingCache<?> iDrawingCache = baseDanmaku.cache;
            if (iDrawingCache == null) {
                return 0;
            }
            if (iDrawingCache.hasReferences()) {
                iDrawingCache.decreaseReference();
                baseDanmaku.cache = null;
                return 0;
            }
            long sizeOf = (long) sizeOf(baseDanmaku);
            iDrawingCache.destroy();
            baseDanmaku.cache = null;
            return sizeOf;
        }

        /* access modifiers changed from: protected */
        public int sizeOf(BaseDanmaku baseDanmaku) {
            if (baseDanmaku.cache == null || baseDanmaku.cache.hasReferences()) {
                return 0;
            }
            return baseDanmaku.cache.size();
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private void clearCachePool() {
            while (true) {
                DrawingCache acquire = this.mCachePool.acquire();
                if (acquire != null) {
                    acquire.destroy();
                } else {
                    return;
                }
            }
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private boolean push(BaseDanmaku baseDanmaku, int i, boolean z) {
            while (true) {
                if (this.mRealSize + i <= this.mMaxSize || this.mCaches.size() <= 0) {
                    break;
                }
                BaseDanmaku first = this.mCaches.first();
                if (first.isTimeOut()) {
                    entryRemoved(false, first, baseDanmaku);
                    this.mCaches.removeItem(first);
                } else if (!z) {
                    return false;
                }
            }
            this.mCaches.addItem(baseDanmaku);
            this.mRealSize += i;
            return true;
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private void clearTimeOutCaches() {
            clearTimeOutCaches(CacheManagingDrawTask.this.mTimer.currMillisecond);
        }

        private void clearTimeOutCaches(long j) {
            IDanmakuIterator it = this.mCaches.iterator();
            while (it.hasNext() && !this.mEndFlag) {
                BaseDanmaku next = it.next();
                if (next.isTimeOut()) {
                    synchronized (CacheManagingDrawTask.this.mDrawingNotify) {
                        try {
                            CacheManagingDrawTask.this.mDrawingNotify.wait(30);
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                            return;
                        }
                    }
                    entryRemoved(false, next, null);
                    it.remove();
                } else {
                    return;
                }
            }
        }

        /* access modifiers changed from: private */
        /* access modifiers changed from: public */
        private BaseDanmaku findReuseableCache(BaseDanmaku baseDanmaku, boolean z, int i) {
            IDanmakuIterator it = this.mCaches.iterator();
            int i2 = 0;
            int slopPixel = !z ? CacheManagingDrawTask.this.mDisp.getSlopPixel() * 2 : 0;
            while (it.hasNext()) {
                int i3 = i2 + 1;
                if (i2 >= i) {
                    return null;
                }
                BaseDanmaku next = it.next();
                IDrawingCache<?> drawingCache = next.getDrawingCache();
                if (!(drawingCache == null || drawingCache.get() == null)) {
                    if (next.paintWidth == baseDanmaku.paintWidth && next.paintHeight == baseDanmaku.paintHeight && next.underlineColor == baseDanmaku.underlineColor && next.borderColor == baseDanmaku.borderColor && next.textColor == baseDanmaku.textColor && next.text.equals(baseDanmaku.text) && next.tag == baseDanmaku.tag) {
                        return next;
                    }
                    if (z) {
                        continue;
                    } else if (!next.isTimeOut()) {
                        return null;
                    } else {
                        if (drawingCache.hasReferences()) {
                            continue;
                        } else {
                            float width = ((float) drawingCache.width()) - baseDanmaku.paintWidth;
                            float height = ((float) drawingCache.height()) - baseDanmaku.paintHeight;
                            if (width >= 0.0f) {
                                float f = (float) slopPixel;
                                if (width <= f && height >= 0.0f && height <= f) {
                                    return next;
                                }
                            } else {
                                continue;
                            }
                        }
                    }
                }
                i2 = i3;
            }
            return null;
        }

        public class CacheHandler extends Handler {
            public static final int ADD_DANMAKKU = 2;
            public static final int BIND_CACHE = 18;
            public static final int BUILD_CACHES = 3;
            public static final int CLEAR_ALL_CACHES = 7;
            public static final int CLEAR_OUTSIDE_CACHES = 8;
            public static final int CLEAR_OUTSIDE_CACHES_AND_RESET = 9;
            public static final int CLEAR_TIMEOUT_CACHES = 4;
            public static final int DISABLE_CANCEL_FLAG = 19;
            public static final int DISPATCH_ACTIONS = 16;
            private static final int PREPARE = 1;
            public static final int QUIT = 6;
            public static final int REBUILD_CACHE = 17;
            public static final int SEEK = 5;
            private boolean mCancelFlag;
            private boolean mIsPlayerPause;
            private boolean mPause;
            private boolean mSeekedFlag;

            public CacheHandler(Looper looper) {
                super(looper);
            }

            public void requestCancelCaching() {
                this.mCancelFlag = true;
            }

            public void handleMessage(Message message) {
                int i = message.what;
                switch (i) {
                    case 1:
                        CacheManager.this.evictAllNotInScreen();
                        for (int i2 = 0; i2 < 300; i2++) {
                            CacheManager.this.mCachePool.release(new DrawingCache());
                        }
                        break;
                    case 2:
                        addDanmakuAndBuildCache((BaseDanmaku) message.obj);
                        return;
                    case 3:
                        removeMessages(3);
                        boolean z = (CacheManagingDrawTask.this.mTaskListener != null && !CacheManagingDrawTask.this.mReadyState) || this.mSeekedFlag;
                        prepareCaches(z);
                        if (z) {
                            this.mSeekedFlag = false;
                        }
                        if (!(CacheManagingDrawTask.this.mTaskListener == null || CacheManagingDrawTask.this.mReadyState)) {
                            CacheManagingDrawTask.this.mTaskListener.ready();
                            CacheManagingDrawTask.this.mReadyState = true;
                            return;
                        }
                        return;
                    case 4:
                        CacheManager.this.clearTimeOutCaches();
                        return;
                    case 5:
                        Long l = (Long) message.obj;
                        if (l != null) {
                            long longValue = l.longValue();
                            long j = CacheManagingDrawTask.this.mCacheTimer.currMillisecond;
                            CacheManagingDrawTask.this.mCacheTimer.update(longValue);
                            this.mSeekedFlag = true;
                            long firstCacheTime = CacheManager.this.getFirstCacheTime();
                            if (longValue > j || firstCacheTime - longValue > CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION) {
                                CacheManager.this.evictAllNotInScreen();
                            } else {
                                CacheManager.this.clearTimeOutCaches();
                            }
                            prepareCaches(true);
                            resume();
                            return;
                        }
                        return;
                    case 6:
                        removeCallbacksAndMessages(null);
                        this.mPause = true;
                        CacheManager.this.evictAll();
                        CacheManager.this.clearCachePool();
                        getLooper().quit();
                        return;
                    case 7:
                        CacheManager.this.evictAll();
                        CacheManagingDrawTask.this.mCacheTimer.update(CacheManagingDrawTask.this.mTimer.currMillisecond - CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION);
                        this.mSeekedFlag = true;
                        return;
                    case 8:
                        CacheManager.this.evictAllNotInScreen(true);
                        CacheManagingDrawTask.this.mCacheTimer.update(CacheManagingDrawTask.this.mTimer.currMillisecond);
                        return;
                    case 9:
                        CacheManager.this.evictAllNotInScreen(true);
                        CacheManagingDrawTask.this.mCacheTimer.update(CacheManagingDrawTask.this.mTimer.currMillisecond);
                        CacheManagingDrawTask.this.requestClear();
                        return;
                    default:
                        switch (i) {
                            case 16:
                                break;
                            case 17:
                                BaseDanmaku baseDanmaku = (BaseDanmaku) message.obj;
                                if (baseDanmaku != null) {
                                    IDrawingCache<?> drawingCache = baseDanmaku.getDrawingCache();
                                    if (!((baseDanmaku.requestFlags & 1) != 0) && drawingCache != null && drawingCache.get() != null && !drawingCache.hasReferences()) {
                                        baseDanmaku.cache = DanmakuUtils.buildDanmakuDrawingCache(baseDanmaku, CacheManagingDrawTask.this.mDisp, (DrawingCache) baseDanmaku.cache);
                                        CacheManager.this.push(baseDanmaku, 0, true);
                                        return;
                                    } else if (baseDanmaku.isLive) {
                                        CacheManager.this.clearCache(baseDanmaku);
                                        createCache(baseDanmaku);
                                        return;
                                    } else {
                                        if (drawingCache != null && drawingCache.hasReferences()) {
                                            drawingCache.destroy();
                                        }
                                        CacheManager.this.entryRemoved(true, baseDanmaku, null);
                                        addDanmakuAndBuildCache(baseDanmaku);
                                        return;
                                    }
                                } else {
                                    return;
                                }
                            case 18:
                                BaseDanmaku baseDanmaku2 = (BaseDanmaku) message.obj;
                                if (!baseDanmaku2.isTimeOut()) {
                                    createCache(baseDanmaku2);
                                    if (baseDanmaku2.cache != null) {
                                        CacheManager.this.push(baseDanmaku2, baseDanmaku2.cache.size(), true);
                                        return;
                                    }
                                    return;
                                }
                                return;
                            case 19:
                                this.mCancelFlag = false;
                                return;
                            default:
                                return;
                        }
                }
                long dispatchAction = dispatchAction();
                if (dispatchAction <= 0) {
                    dispatchAction = CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION / 2;
                }
                sendEmptyMessageDelayed(16, dispatchAction);
            }

            private long dispatchAction() {
                if (CacheManagingDrawTask.this.mCacheTimer.currMillisecond <= CacheManagingDrawTask.this.mTimer.currMillisecond - CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION) {
                    CacheManager.this.evictAllNotInScreen();
                    CacheManagingDrawTask.this.mCacheTimer.update(CacheManagingDrawTask.this.mTimer.currMillisecond);
                    sendEmptyMessage(3);
                    return 0;
                }
                float poolPercent = CacheManager.this.getPoolPercent();
                BaseDanmaku first = CacheManager.this.mCaches.first();
                long actualTime = first != null ? first.getActualTime() - CacheManagingDrawTask.this.mTimer.currMillisecond : 0;
                long j = CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION * 2;
                if (poolPercent < 0.6f && actualTime > CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION) {
                    CacheManagingDrawTask.this.mCacheTimer.update(CacheManagingDrawTask.this.mTimer.currMillisecond);
                    removeMessages(3);
                    sendEmptyMessage(3);
                    return 0;
                } else if (poolPercent > 0.4f && actualTime < (-j)) {
                    removeMessages(4);
                    sendEmptyMessage(4);
                    return 0;
                } else if (poolPercent >= 0.9f) {
                    return 0;
                } else {
                    long j2 = CacheManagingDrawTask.this.mCacheTimer.currMillisecond - CacheManagingDrawTask.this.mTimer.currMillisecond;
                    if (first != null && first.isTimeOut() && j2 < (-CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION)) {
                        CacheManagingDrawTask.this.mCacheTimer.update(CacheManagingDrawTask.this.mTimer.currMillisecond);
                        sendEmptyMessage(8);
                        sendEmptyMessage(3);
                        return 0;
                    } else if (j2 > j) {
                        return 0;
                    } else {
                        removeMessages(3);
                        sendEmptyMessage(3);
                        return 0;
                    }
                }
            }

            private void releaseDanmakuCache(BaseDanmaku baseDanmaku, DrawingCache drawingCache) {
                if (drawingCache == null) {
                    drawingCache = (DrawingCache) baseDanmaku.cache;
                }
                baseDanmaku.cache = null;
                if (drawingCache != null) {
                    drawingCache.destroy();
                    CacheManager.this.mCachePool.release(drawingCache);
                }
            }

            private void preMeasure() {
                IDanmakus iDanmakus;
                try {
                    long j = CacheManagingDrawTask.this.mTimer.currMillisecond;
                    iDanmakus = CacheManagingDrawTask.this.danmakuList.subnew(j - CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION, (CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION * 2) + j);
                } catch (Exception unused) {
                    iDanmakus = null;
                }
                if (iDanmakus != null && !iDanmakus.isEmpty()) {
                    IDanmakuIterator it = iDanmakus.iterator();
                    while (it.hasNext() && !this.mPause) {
                        BaseDanmaku next = it.next();
                        if (!next.hasPassedFilter()) {
                            CacheManagingDrawTask.this.mContext.mDanmakuFilters.filter(next, 0, 0, null, true, CacheManagingDrawTask.this.mContext);
                        }
                        if (!next.isFiltered()) {
                            if (!next.isMeasured()) {
                                next.measure(CacheManagingDrawTask.this.mDisp, true);
                            }
                            if (!next.isPrepared()) {
                                next.prepare(CacheManagingDrawTask.this.mDisp, true);
                            }
                        }
                    }
                }
            }

            /* JADX WARNING: Removed duplicated region for block: B:14:0x0055  */
            /* JADX WARNING: Removed duplicated region for block: B:16:0x0061  */
            private long prepareCaches(boolean z) {
                CacheHandler cacheHandler;
                BaseDanmaku next;
                preMeasure();
                long j = CacheManagingDrawTask.this.mCacheTimer.currMillisecond;
                long j2 = (CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION * ((long) CacheManager.this.mScreenSize)) + j;
                long j3 = 0;
                if (j2 < CacheManagingDrawTask.this.mTimer.currMillisecond) {
                    return 0;
                }
                long uptimeMillis = SystemClock.uptimeMillis();
                IDanmakus iDanmakus = null;
                int i = 0;
                boolean z2 = false;
                do {
                    try {
                        iDanmakus = CacheManagingDrawTask.this.danmakuList.subnew(j, j2);
                    } catch (Exception unused) {
                        SystemClock.sleep(10);
                        z2 = true;
                    }
                    i++;
                    if (i >= 3 || iDanmakus != null) {
                        if (iDanmakus != null) {
                            CacheManagingDrawTask.this.mCacheTimer.update(j2);
                            return 0;
                        }
                        BaseDanmaku first = iDanmakus.first();
                        BaseDanmaku last = iDanmakus.last();
                        if (first == null || last == null) {
                            CacheManagingDrawTask.this.mCacheTimer.update(j2);
                            return 0;
                        }
                        long actualTime = first.getActualTime() - CacheManagingDrawTask.this.mTimer.currMillisecond;
                        long j4 = 30;
                        if (actualTime < 0) {
                            cacheHandler = this;
                        } else {
                            cacheHandler = this;
                            j4 = ((actualTime * 10) / CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION) + 30;
                        }
                        long min = Math.min(100L, j4);
                        if (!z) {
                            j3 = min;
                        }
                        IDanmakuIterator it = iDanmakus.iterator();
                        int size = iDanmakus.size();
                        int i2 = 0;
                        int i3 = 0;
                        BaseDanmaku baseDanmaku = null;
                        while (true) {
                            if (cacheHandler.mPause || cacheHandler.mCancelFlag || !it.hasNext()) {
                                break;
                            }
                            next = it.next();
                            if (last.getActualTime() < CacheManagingDrawTask.this.mTimer.currMillisecond) {
                                break;
                            }
                            IDrawingCache<?> drawingCache = next.getDrawingCache();
                            if ((drawingCache == null || drawingCache.get() == null) && (z || (!next.isTimeOut() && next.isOutside()))) {
                                if (!next.hasPassedFilter()) {
                                    CacheManagingDrawTask.this.mContext.mDanmakuFilters.filter(next, i2, size, null, true, CacheManagingDrawTask.this.mContext);
                                }
                                if (next.priority != 0 || !next.isFiltered()) {
                                    if (next.getType() == 1) {
                                        int actualTime2 = (int) ((next.getActualTime() - j) / CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION);
                                        i3 = i3;
                                        if (i3 == actualTime2) {
                                            i2++;
                                        } else {
                                            i3 = actualTime2;
                                            i2 = 0;
                                        }
                                    }
                                    if (!z && !cacheHandler.mIsPlayerPause) {
                                        try {
                                            synchronized (CacheManagingDrawTask.this.mDrawingNotify) {
                                                CacheManagingDrawTask.this.mDrawingNotify.wait(j3);
                                            }
                                        } catch (InterruptedException e) {
                                            e.printStackTrace();
                                        }
                                    }
                                    if (cacheHandler.buildCache(next, false) == 1) {
                                        break;
                                    } else if (!z) {
                                        DanmakuFactory danmakuFactory = CacheManagingDrawTask.this.mContext.mDanmakuFactory;
                                        if (SystemClock.uptimeMillis() - uptimeMillis >= ((long) CacheManager.this.mScreenSize) * DanmakuFactory.COMMON_DANMAKU_DURATION) {
                                            break;
                                        }
                                    } else {
                                        continue;
                                    }
                                }
                            }
                            baseDanmaku = next;
                            last = last;
                        }
                        baseDanmaku = next;
                        long uptimeMillis2 = SystemClock.uptimeMillis() - uptimeMillis;
                        if (baseDanmaku != null) {
                            CacheManagingDrawTask.this.mCacheTimer.update(baseDanmaku.getActualTime());
                        } else {
                            CacheManagingDrawTask.this.mCacheTimer.update(j2);
                        }
                        return uptimeMillis2;
                    }
                } while (z2);
                if (iDanmakus != null) {
                }
            }

            /* JADX WARNING: Removed duplicated region for block: B:14:0x0030  */
            /* JADX WARNING: Removed duplicated region for block: B:20:0x003d  */
            public boolean createCache(BaseDanmaku baseDanmaku) {
                DrawingCache drawingCache;
                if (!baseDanmaku.isMeasured()) {
                    baseDanmaku.measure(CacheManagingDrawTask.this.mDisp, true);
                }
                try {
                    try {
                        drawingCache = DanmakuUtils.buildDanmakuDrawingCache(baseDanmaku, CacheManagingDrawTask.this.mDisp, CacheManager.this.mCachePool.acquire());
                        baseDanmaku.cache = drawingCache;
                        return true;
                    } catch (OutOfMemoryError unused) {
                        if (drawingCache != null) {
                        }
                        baseDanmaku.cache = null;
                        return false;
                    } catch (Exception unused2) {
                        if (drawingCache != null) {
                        }
                        baseDanmaku.cache = null;
                        return false;
                    }
                } catch (OutOfMemoryError unused3) {
                    drawingCache = null;
                    if (drawingCache != null) {
                        CacheManager.this.mCachePool.release(drawingCache);
                    }
                    baseDanmaku.cache = null;
                    return false;
                } catch (Exception unused4) {
                    drawingCache = null;
                    if (drawingCache != null) {
                        CacheManager.this.mCachePool.release(drawingCache);
                    }
                    baseDanmaku.cache = null;
                    return false;
                }
            }

            private byte buildCache(BaseDanmaku baseDanmaku, boolean z) {
                if (!baseDanmaku.isMeasured()) {
                    baseDanmaku.measure(CacheManagingDrawTask.this.mDisp, true);
                }
                DrawingCache drawingCache = null;
                try {
                    BaseDanmaku findReuseableCache = CacheManager.this.findReuseableCache(baseDanmaku, true, 20);
                    DrawingCache drawingCache2 = findReuseableCache != null ? (DrawingCache) findReuseableCache.cache : null;
                    if (drawingCache2 != null) {
                        try {
                            drawingCache2.increaseReference();
                            baseDanmaku.cache = drawingCache2;
                            CacheManagingDrawTask.this.mCacheManager.push(baseDanmaku, 0, z);
                            return 0;
                        } catch (OutOfMemoryError unused) {
                            drawingCache = drawingCache2;
                            releaseDanmakuCache(baseDanmaku, drawingCache);
                            return 1;
                        } catch (Exception unused2) {
                            drawingCache = drawingCache2;
                            releaseDanmakuCache(baseDanmaku, drawingCache);
                            return 1;
                        }
                    } else {
                        BaseDanmaku findReuseableCache2 = CacheManager.this.findReuseableCache(baseDanmaku, false, 50);
                        if (findReuseableCache2 != null) {
                            drawingCache2 = (DrawingCache) findReuseableCache2.cache;
                        }
                        if (drawingCache2 != null) {
                            findReuseableCache2.cache = null;
                            baseDanmaku.cache = DanmakuUtils.buildDanmakuDrawingCache(baseDanmaku, CacheManagingDrawTask.this.mDisp, drawingCache2);
                            CacheManagingDrawTask.this.mCacheManager.push(baseDanmaku, 0, z);
                            return 0;
                        }
                        if (!z) {
                            if (CacheManager.this.mRealSize + DanmakuUtils.getCacheSize((int) baseDanmaku.paintWidth, (int) baseDanmaku.paintHeight) > CacheManager.this.mMaxSize) {
                                return 1;
                            }
                        }
                        DrawingCache buildDanmakuDrawingCache = DanmakuUtils.buildDanmakuDrawingCache(baseDanmaku, CacheManagingDrawTask.this.mDisp, CacheManager.this.mCachePool.acquire());
                        baseDanmaku.cache = buildDanmakuDrawingCache;
                        boolean push = CacheManagingDrawTask.this.mCacheManager.push(baseDanmaku, CacheManager.this.sizeOf(baseDanmaku), z);
                        if (!push) {
                            releaseDanmakuCache(baseDanmaku, buildDanmakuDrawingCache);
                        }
                        return !push ? 1 : 0 ? (byte) 1 : 0;
                    }
                } catch (OutOfMemoryError unused3) {
                    releaseDanmakuCache(baseDanmaku, drawingCache);
                    return 1;
                } catch (Exception unused4) {
                    releaseDanmakuCache(baseDanmaku, drawingCache);
                    return 1;
                }
            }

            private final void addDanmakuAndBuildCache(BaseDanmaku baseDanmaku) {
                if (baseDanmaku.isTimeOut()) {
                    return;
                }
                if (baseDanmaku.getActualTime() > CacheManagingDrawTask.this.mCacheTimer.currMillisecond + CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION && !baseDanmaku.isLive) {
                    return;
                }
                if (baseDanmaku.priority != 0 || !baseDanmaku.isFiltered()) {
                    IDrawingCache<?> drawingCache = baseDanmaku.getDrawingCache();
                    if (drawingCache == null || drawingCache.get() == null) {
                        buildCache(baseDanmaku, true);
                    }
                }
            }

            public void begin() {
                sendEmptyMessage(1);
                sendEmptyMessageDelayed(4, CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION);
            }

            public void pause() {
                this.mPause = true;
                removeCallbacksAndMessages(null);
                sendEmptyMessage(6);
            }

            public void resume() {
                sendEmptyMessage(19);
                this.mPause = false;
                removeMessages(16);
                sendEmptyMessage(16);
                sendEmptyMessageDelayed(4, CacheManagingDrawTask.this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION);
            }

            public boolean isPause() {
                return this.mPause;
            }

            public void requestBuildCacheAndDraw(long j) {
                removeMessages(3);
                this.mSeekedFlag = true;
                sendEmptyMessage(19);
                CacheManagingDrawTask.this.mCacheTimer.update(CacheManagingDrawTask.this.mTimer.currMillisecond + j);
                sendEmptyMessage(3);
            }

            public void onPlayStateChanged(boolean z) {
                this.mIsPlayerPause = !z;
            }
        }

        public long getFirstCacheTime() {
            BaseDanmaku first;
            Danmakus danmakus = this.mCaches;
            if (danmakus == null || danmakus.size() <= 0 || (first = this.mCaches.first()) == null) {
                return 0;
            }
            return first.getActualTime();
        }

        public void requestBuild(long j) {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.requestBuildCacheAndDraw(j);
            }
        }

        public void requestClearAll() {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.removeMessages(3);
                this.mHandler.removeMessages(19);
                this.mHandler.requestCancelCaching();
                this.mHandler.removeMessages(7);
                this.mHandler.sendEmptyMessage(7);
            }
        }

        public void requestClearUnused() {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.removeMessages(9);
                this.mHandler.sendEmptyMessage(9);
            }
        }

        public void requestClearTimeout() {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.removeMessages(4);
                this.mHandler.sendEmptyMessage(4);
            }
        }

        public void post(Runnable runnable) {
            CacheHandler cacheHandler = this.mHandler;
            if (cacheHandler != null) {
                cacheHandler.post(runnable);
            }
        }
    }

    @Override // master.flame.danmaku.controller.DrawTask
    public boolean onDanmakuConfigChanged(DanmakuContext danmakuContext, DanmakuContext.DanmakuConfigTag danmakuConfigTag, Object... objArr) {
        CacheManager cacheManager;
        CacheManager cacheManager2;
        if (!super.handleOnDanmakuConfigChanged(danmakuContext, danmakuConfigTag, objArr)) {
            if (DanmakuContext.DanmakuConfigTag.SCROLL_SPEED_FACTOR.equals(danmakuConfigTag)) {
                this.mDisp.resetSlopPixel(this.mContext.scaleTextSize);
                requestClear();
            } else if (danmakuConfigTag.isVisibilityRelatedTag()) {
                if (objArr != null && objArr.length > 0 && objArr[0] != null && ((!(objArr[0] instanceof Boolean) || ((Boolean) objArr[0]).booleanValue()) && (cacheManager2 = this.mCacheManager) != null)) {
                    cacheManager2.requestBuild(0);
                }
                requestClear();
            } else if (DanmakuContext.DanmakuConfigTag.TRANSPARENCY.equals(danmakuConfigTag) || DanmakuContext.DanmakuConfigTag.SCALE_TEXTSIZE.equals(danmakuConfigTag) || DanmakuContext.DanmakuConfigTag.DANMAKU_STYLE.equals(danmakuConfigTag)) {
                if (DanmakuContext.DanmakuConfigTag.SCALE_TEXTSIZE.equals(danmakuConfigTag)) {
                    this.mDisp.resetSlopPixel(this.mContext.scaleTextSize);
                }
                CacheManager cacheManager3 = this.mCacheManager;
                if (cacheManager3 != null) {
                    cacheManager3.requestClearAll();
                    this.mCacheManager.requestBuild(-this.mContext.mDanmakuFactory.MAX_DANMAKU_DURATION);
                }
            } else {
                CacheManager cacheManager4 = this.mCacheManager;
                if (cacheManager4 != null) {
                    cacheManager4.requestClearUnused();
                    this.mCacheManager.requestBuild(0);
                }
            }
        }
        if (this.mTaskListener == null || (cacheManager = this.mCacheManager) == null) {
            return true;
        }
        cacheManager.post(new Runnable() {
            /* class master.flame.danmaku.controller.CacheManagingDrawTask.AnonymousClass1 */

            public void run() {
                CacheManagingDrawTask.this.mTaskListener.onDanmakuConfigChanged();
            }
        });
        return true;
    }
}
