package com.taobao.weex.layout.measurefunc;

import android.graphics.Canvas;
import android.os.Build;
import android.os.Looper;
import android.text.Editable;
import android.text.Layout;
import android.text.Spannable;
import android.text.SpannableString;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.SpannedString;
import android.text.StaticLayout;
import android.text.TextPaint;
import android.text.TextUtils;
import android.text.style.AlignmentSpan;
import com.taobao.weex.WXSDKManager;
import com.taobao.weex.common.Constants;
import com.taobao.weex.dom.TextDecorationSpan;
import com.taobao.weex.dom.WXAttr;
import com.taobao.weex.dom.WXCustomStyleSpan;
import com.taobao.weex.dom.WXLineHeightSpan;
import com.taobao.weex.dom.WXStyle;
import com.taobao.weex.layout.ContentBoxMeasurement;
import com.taobao.weex.layout.MeasureMode;
import com.taobao.weex.ui.component.WXComponent;
import com.taobao.weex.ui.component.WXTextDecoration;
import com.taobao.weex.utils.WXDomUtils;
import com.taobao.weex.utils.WXLogUtils;
import com.taobao.weex.utils.WXResourceUtils;
import com.taobao.weex.utils.WXUtils;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

public class TextContentBoxMeasurement extends ContentBoxMeasurement {
    private static final Canvas DUMMY_CANVAS = new Canvas();
    private static final String ELLIPSIS = "…";
    private AtomicReference<Layout> atomicReference = new AtomicReference<>();
    private boolean hasBeenMeasured = false;
    private Layout layout;
    protected Layout.Alignment mAlignment;
    protected int mColor;
    protected String mFontFamily = null;
    protected int mFontSize = -1;
    protected int mFontStyle = -1;
    protected int mFontWeight = -1;
    protected boolean mIsColorSet = false;
    protected int mLineHeight = -1;
    private int mNumberOfLines = -1;
    private String mText = null;
    protected WXTextDecoration mTextDecoration = WXTextDecoration.NONE;
    protected TextPaint mTextPaint;
    private float previousWidth = Float.NaN;
    private Spanned spanned;
    private TextUtils.TruncateAt textOverflow;

    public TextContentBoxMeasurement(WXComponent wXComponent) {
        super(wXComponent);
    }

    class SetSpanOperation {
        protected final int end;
        protected final int flag;
        protected final int start;
        protected final Object what;

        SetSpanOperation(TextContentBoxMeasurement textContentBoxMeasurement, int i, int i2, Object obj) {
            this(i, i2, obj, 17);
        }

        SetSpanOperation(int i, int i2, Object obj, int i3) {
            this.start = i;
            this.end = i2;
            this.what = obj;
            this.flag = i3;
        }

        public void execute(Spannable spannable) {
            spannable.setSpan(this.what, this.start, this.end, this.flag);
        }
    }

    @Override // com.taobao.weex.layout.ContentBoxMeasurement
    public void layoutBefore() {
        this.mTextPaint = new TextPaint(1);
        this.hasBeenMeasured = false;
        updateStyleAndText();
        this.spanned = createSpanned(this.mText);
    }

    @Override // com.taobao.weex.layout.ContentBoxMeasurement
    public void measureInternal(float f, float f2, int i, int i2) {
        boolean z = true;
        this.hasBeenMeasured = true;
        TextPaint textPaint = this.mTextPaint;
        if (i != MeasureMode.EXACTLY) {
            z = false;
        }
        float textWidth = getTextWidth(textPaint, f, z);
        if (textWidth <= 0.0f || this.spanned == null) {
            if (i == MeasureMode.UNSPECIFIED) {
                f = 0.0f;
            }
            if (i2 == MeasureMode.UNSPECIFIED) {
                f2 = 0.0f;
            }
        } else {
            Layout createLayout = createLayout(textWidth, null);
            this.layout = createLayout;
            this.previousWidth = (float) createLayout.getWidth();
            if (Float.isNaN(f)) {
                f = (float) this.layout.getWidth();
            } else {
                f = Math.min((float) this.layout.getWidth(), f);
            }
            if (Float.isNaN(f2)) {
                f2 = (float) this.layout.getHeight();
            }
        }
        this.mMeasureWidth = f;
        this.mMeasureHeight = f2;
    }

    @Override // com.taobao.weex.layout.ContentBoxMeasurement
    public void layoutAfter(float f, float f2) {
        if (this.mComponent != null) {
            if (!this.hasBeenMeasured) {
                updateStyleAndText();
                recalculateLayout(f);
            } else if (!(this.layout == null || WXDomUtils.getContentWidth(this.mComponent.getPadding(), this.mComponent.getBorder(), f) == this.previousWidth)) {
                recalculateLayout(f);
            }
            this.hasBeenMeasured = false;
            Layout layout2 = this.layout;
            if (layout2 != null && !layout2.equals(this.atomicReference.get()) && Build.VERSION.SDK_INT >= 19 && Thread.currentThread() != Looper.getMainLooper().getThread()) {
                warmUpTextLayoutCache(this.layout);
            }
            swap();
            WXSDKManager.getInstance().getWXRenderManager().postOnUiThread(new Runnable() {
                /* class com.taobao.weex.layout.measurefunc.TextContentBoxMeasurement.AnonymousClass1 */

                public void run() {
                    if (TextContentBoxMeasurement.this.mComponent != null) {
                        TextContentBoxMeasurement.this.mComponent.updateExtra(TextContentBoxMeasurement.this.atomicReference.get());
                    }
                }
            }, this.mComponent.getInstanceId());
        }
    }

    private void updateStyleAndText() {
        updateStyleImp(this.mComponent.getStyles());
        this.mText = WXAttr.getValue(this.mComponent.getAttrs());
    }

    public void forceRelayout() {
        layoutBefore();
        measure(this.previousWidth, Float.NaN, MeasureMode.EXACTLY, MeasureMode.UNSPECIFIED);
        layoutAfter(this.previousWidth, Float.NaN);
    }

    private void updateStyleImp(Map<String, Object> map) {
        if (map != null) {
            if (map.containsKey(Constants.Name.LINES)) {
                int lines = WXStyle.getLines(map);
                if (lines <= 0) {
                    lines = -1;
                }
                this.mNumberOfLines = lines;
            }
            if (map.containsKey(Constants.Name.FONT_SIZE)) {
                this.mFontSize = WXStyle.getFontSize(map, this.mComponent.getInstance().getDefaultFontSize(), this.mComponent.getViewPortWidthForFloat());
            }
            if (map.containsKey(Constants.Name.FONT_WEIGHT)) {
                this.mFontWeight = WXStyle.getFontWeight(map);
            }
            if (map.containsKey(Constants.Name.FONT_STYLE)) {
                this.mFontStyle = WXStyle.getFontStyle(map);
            }
            if (map.containsKey("color")) {
                int color = WXResourceUtils.getColor(WXStyle.getTextColor(map));
                this.mColor = color;
                this.mIsColorSet = color != Integer.MIN_VALUE;
            }
            if (map.containsKey(Constants.Name.TEXT_DECORATION)) {
                this.mTextDecoration = WXStyle.getTextDecoration(map);
            }
            if (map.containsKey(Constants.Name.FONT_FAMILY)) {
                this.mFontFamily = WXStyle.getFontFamily(map);
            }
            this.mAlignment = WXStyle.getTextAlignment(map, this.mComponent.isLayoutRTL());
            this.textOverflow = WXStyle.getTextOverflow(map);
            int lineHeight = WXStyle.getLineHeight(map, this.mComponent.getViewPortWidthForFloat());
            if (lineHeight != -1) {
                this.mLineHeight = lineHeight;
            }
        }
    }

    /* access modifiers changed from: protected */
    public Spanned createSpanned(String str) {
        if (TextUtils.isEmpty(str)) {
            return new SpannableString("");
        }
        SpannableString spannableString = new SpannableString(str);
        updateSpannable(spannableString, 17);
        return spannableString;
    }

    /* access modifiers changed from: protected */
    public void updateSpannable(Spannable spannable, int i) {
        int length = spannable.length();
        int i2 = this.mFontSize;
        if (i2 == -1) {
            this.mTextPaint.setTextSize((float) this.mComponent.getInstance().getDefaultFontSize());
        } else {
            this.mTextPaint.setTextSize((float) i2);
        }
        int i3 = this.mLineHeight;
        if (i3 != -1) {
            setSpan(spannable, new WXLineHeightSpan(i3), 0, length, i);
        }
        setSpan(spannable, new AlignmentSpan.Standard(this.mAlignment), 0, length, i);
        if (!(this.mFontStyle == -1 && this.mFontWeight == -1 && this.mFontFamily == null)) {
            setSpan(spannable, new WXCustomStyleSpan(this.mFontStyle, this.mFontWeight, this.mFontFamily), 0, length, i);
        }
        if (this.mIsColorSet) {
            this.mTextPaint.setColor(this.mColor);
        }
        if (this.mTextDecoration == WXTextDecoration.UNDERLINE || this.mTextDecoration == WXTextDecoration.LINETHROUGH) {
            setSpan(spannable, new TextDecorationSpan(this.mTextDecoration), 0, length, i);
        }
    }

    /* access modifiers changed from: protected */
    public void setSpan(Spannable spannable, Object obj, int i, int i2, int i3) {
        spannable.setSpan(obj, i, i2, i3);
    }

    private float getTextWidth(TextPaint textPaint, float f, boolean z) {
        if (this.mText == null) {
            if (z) {
                return f;
            }
            return 0.0f;
        } else if (z) {
            return f;
        } else {
            float desiredWidth = Layout.getDesiredWidth(this.spanned, textPaint);
            return (WXUtils.isUndefined(f) || desiredWidth < f) ? desiredWidth : f;
        }
    }

    private Layout createLayout(float f, Layout layout2) {
        SpannableStringBuilder spannableStringBuilder;
        if (this.previousWidth != f || layout2 == null) {
            layout2 = new StaticLayout(this.spanned, this.mTextPaint, (int) Math.ceil((double) f), Layout.Alignment.ALIGN_NORMAL, 1.0f, 0.0f, false);
        }
        int i = this.mNumberOfLines;
        if (i != -1 && i > 0 && i < layout2.getLineCount()) {
            int i2 = this.mNumberOfLines;
            if (i2 > 1) {
                int lineStart = layout2.getLineStart(i2 - 1);
                int lineEnd = layout2.getLineEnd(this.mNumberOfLines - 1);
                if (lineStart < lineEnd) {
                    if (lineStart > 0) {
                        spannableStringBuilder = new SpannableStringBuilder(this.spanned.subSequence(0, lineStart));
                    } else {
                        spannableStringBuilder = new SpannableStringBuilder();
                    }
                    double d = (double) f;
                    spannableStringBuilder.append((CharSequence) truncate(new SpannableStringBuilder(this.spanned.subSequence(lineStart, lineEnd)), this.mTextPaint, (int) Math.ceil(d), this.textOverflow));
                    adjustSpansRange(this.spanned, spannableStringBuilder);
                    this.spanned = spannableStringBuilder;
                    return new StaticLayout(this.spanned, this.mTextPaint, (int) Math.ceil(d), Layout.Alignment.ALIGN_NORMAL, 1.0f, 0.0f, false);
                }
            } else {
                int ceil = (int) Math.ceil((double) f);
                Spanned spanned2 = this.spanned;
                return StaticLayout.Builder.obtain(spanned2, 0, spanned2.length(), this.mTextPaint, ceil).setMaxLines(1).setEllipsize(TextUtils.TruncateAt.END).setEllipsizedWidth(ceil).build();
            }
        }
        return layout2;
    }

    private Spanned truncate(Editable editable, TextPaint textPaint, int i, TextUtils.TruncateAt truncateAt) {
        SpannedString spannedString = new SpannedString("");
        if (!TextUtils.isEmpty(editable) && editable.length() > 0) {
            if (truncateAt != null) {
                editable.append(ELLIPSIS);
                Object[] spans = editable.getSpans(0, editable.length(), Object.class);
                for (Object obj : spans) {
                    int spanStart = editable.getSpanStart(obj);
                    int spanEnd = editable.getSpanEnd(obj);
                    if (spanStart == 0 && spanEnd == editable.length() - 1) {
                        editable.removeSpan(obj);
                        editable.setSpan(obj, 0, editable.length(), editable.getSpanFlags(obj));
                    }
                }
            }
            while (editable.length() > 1) {
                int length = editable.length() - 1;
                if (truncateAt != null) {
                    length--;
                }
                editable.delete(length, length + 1);
                if (new StaticLayout(editable, textPaint, i, Layout.Alignment.ALIGN_NORMAL, 1.0f, 0.0f, false).getLineCount() <= 1) {
                    return editable;
                }
            }
        }
        return spannedString;
    }

    private void adjustSpansRange(Spanned spanned2, Spannable spannable) {
        Object[] spans = spanned2.getSpans(0, spanned2.length(), Object.class);
        for (Object obj : spans) {
            int spanStart = spanned2.getSpanStart(obj);
            int spanEnd = spanned2.getSpanEnd(obj);
            if (spanStart == 0 && spanEnd == spanned2.length()) {
                spannable.removeSpan(obj);
                spannable.setSpan(obj, 0, spannable.length(), spanned2.getSpanFlags(obj));
            }
        }
    }

    private void recalculateLayout(float f) {
        float contentWidth = WXDomUtils.getContentWidth(this.mComponent.getPadding(), this.mComponent.getBorder(), f);
        if (contentWidth > 0.0f) {
            Spanned createSpanned = createSpanned(this.mText);
            this.spanned = createSpanned;
            if (createSpanned != null) {
                Layout createLayout = createLayout(contentWidth, this.layout);
                this.layout = createLayout;
                this.previousWidth = (float) createLayout.getWidth();
                return;
            }
            this.previousWidth = 0.0f;
        }
    }

    private boolean warmUpTextLayoutCache(Layout layout2) {
        try {
            layout2.draw(DUMMY_CANVAS);
            return true;
        } catch (Exception e) {
            WXLogUtils.eTag("TextWarmUp", e);
            return false;
        }
    }

    private void swap() {
        Layout layout2 = this.layout;
        if (layout2 != null) {
            this.atomicReference.set(layout2);
            this.layout = null;
        }
        this.hasBeenMeasured = false;
    }
}
