package com.alibaba.fastjson;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

public abstract class JSONValidator implements Cloneable, Closeable {
    protected char ch;
    protected int count = 0;
    protected boolean eof;
    protected int pos = -1;
    protected boolean supportMultiValue = false;
    protected Type type;
    private Boolean validateResult;

    public enum Type {
        Object,
        Array,
        Value
    }

    static final boolean isWhiteSpace(char c) {
        return c == ' ' || c == '\t' || c == '\r' || c == '\n' || c == '\f' || c == '\b';
    }

    @Override // java.io.Closeable, java.lang.AutoCloseable
    public void close() throws IOException {
    }

    /* access modifiers changed from: package-private */
    public abstract void next();

    public static JSONValidator fromUtf8(byte[] bArr) {
        return new UTF8Validator(bArr);
    }

    public static JSONValidator fromUtf8(InputStream inputStream) {
        return new UTF8InputStreamValidator(inputStream);
    }

    public static JSONValidator from(String str) {
        return new UTF16Validator(str);
    }

    public static JSONValidator from(Reader reader) {
        return new ReaderValidator(reader);
    }

    public boolean isSupportMultiValue() {
        return this.supportMultiValue;
    }

    public JSONValidator setSupportMultiValue(boolean z) {
        this.supportMultiValue = z;
        return this;
    }

    public Type getType() {
        if (this.type == null) {
            validate();
        }
        return this.type;
    }

    public boolean validate() {
        Boolean bool = this.validateResult;
        if (bool != null) {
            return bool.booleanValue();
        }
        while (any()) {
            skipWhiteSpace();
            this.count++;
            if (this.eof) {
                this.validateResult = true;
                return true;
            } else if (this.supportMultiValue) {
                skipWhiteSpace();
                if (this.eof) {
                    this.validateResult = true;
                    return true;
                }
            } else {
                this.validateResult = false;
                return false;
            }
        }
        this.validateResult = false;
        return false;
    }

    /* JADX WARNING: Code restructure failed: missing block: B:118:0x0173, code lost:
        if (r0 <= '9') goto L_0x0176;
     */
    /* JADX WARNING: Removed duplicated region for block: B:124:0x0183  */
    private boolean any() {
        char c;
        char c2;
        char c3;
        char c4;
        char c5 = this.ch;
        if (c5 != '\"') {
            if (!(c5 == '+' || c5 == '-')) {
                if (c5 == '[') {
                    next();
                    skipWhiteSpace();
                    if (this.ch == ']') {
                        next();
                        this.type = Type.Array;
                        return true;
                    }
                    while (any()) {
                        skipWhiteSpace();
                        char c6 = this.ch;
                        if (c6 == ',') {
                            next();
                            skipWhiteSpace();
                        } else if (c6 != ']') {
                            return false;
                        } else {
                            next();
                            this.type = Type.Array;
                            return true;
                        }
                    }
                    return false;
                } else if (c5 == 'f') {
                    next();
                    if (this.ch != 'a') {
                        return false;
                    }
                    next();
                    if (this.ch != 'l') {
                        return false;
                    }
                    next();
                    if (this.ch != 's') {
                        return false;
                    }
                    next();
                    if (this.ch != 'e') {
                        return false;
                    }
                    next();
                    if (!isWhiteSpace(this.ch) && (c2 = this.ch) != ',' && c2 != ']' && c2 != '}' && c2 != 0) {
                        return false;
                    }
                    this.type = Type.Value;
                    return true;
                } else if (c5 == 'n') {
                    next();
                    if (this.ch != 'u') {
                        return false;
                    }
                    next();
                    if (this.ch != 'l') {
                        return false;
                    }
                    next();
                    if (this.ch != 'l') {
                        return false;
                    }
                    next();
                    if (!isWhiteSpace(this.ch) && (c3 = this.ch) != ',' && c3 != ']' && c3 != '}' && c3 != 0) {
                        return false;
                    }
                    this.type = Type.Value;
                    return true;
                } else if (c5 == 't') {
                    next();
                    if (this.ch != 'r') {
                        return false;
                    }
                    next();
                    if (this.ch != 'u') {
                        return false;
                    }
                    next();
                    if (this.ch != 'e') {
                        return false;
                    }
                    next();
                    if (!isWhiteSpace(this.ch) && (c4 = this.ch) != ',' && c4 != ']' && c4 != '}' && c4 != 0) {
                        return false;
                    }
                    this.type = Type.Value;
                    return true;
                } else if (c5 != '{') {
                    switch (c5) {
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            break;
                        default:
                            return false;
                    }
                } else {
                    next();
                    while (isWhiteSpace(this.ch)) {
                        next();
                    }
                    if (this.ch == '}') {
                        next();
                        this.type = Type.Object;
                        return true;
                    }
                    while (true) {
                        if (this.ch != '\"') {
                            break;
                        }
                        fieldName();
                        skipWhiteSpace();
                        if (this.ch != ':') {
                            break;
                        }
                        next();
                        skipWhiteSpace();
                        if (!any()) {
                            return false;
                        }
                        skipWhiteSpace();
                        char c7 = this.ch;
                        if (c7 == ',') {
                            next();
                            skipWhiteSpace();
                        } else if (c7 == '}') {
                            next();
                            this.type = Type.Object;
                            return true;
                        }
                    }
                    return false;
                }
            }
            if (c5 == '-' || c5 == '+') {
                next();
                skipWhiteSpace();
                char c8 = this.ch;
                if (c8 >= '0') {
                }
                return false;
            }
            do {
                next();
                c = this.ch;
                if (c >= '0') {
                }
                if (c == '.') {
                    next();
                    char c9 = this.ch;
                    if (c9 < '0' || c9 > '9') {
                        return false;
                    }
                    while (true) {
                        char c10 = this.ch;
                        if (c10 >= '0' && c10 <= '9') {
                            next();
                        }
                    }
                }
                char c11 = this.ch;
                if (c11 == 'e' || c11 == 'E') {
                    next();
                    char c12 = this.ch;
                    if (c12 == '-' || c12 == '+') {
                        next();
                    }
                    char c13 = this.ch;
                    if (c13 >= '0' && c13 <= '9') {
                        next();
                        while (true) {
                            char c14 = this.ch;
                            if (c14 >= '0' && c14 <= '9') {
                                next();
                            }
                        }
                    }
                    return false;
                }
                this.type = Type.Value;
                return true;
            } while (c <= '9');
            if (c == '.') {
            }
            char c112 = this.ch;
            next();
            char c122 = this.ch;
            next();
            char c132 = this.ch;
            next();
            while (true) {
                char c142 = this.ch;
                next();
            }
            this.type = Type.Value;
            return true;
        }
        next();
        while (!this.eof) {
            char c15 = this.ch;
            if (c15 == '\\') {
                next();
                if (this.ch == 'u') {
                    next();
                    next();
                    next();
                    next();
                    next();
                } else {
                    next();
                }
            } else if (c15 == '\"') {
                next();
                this.type = Type.Value;
                return true;
            } else {
                next();
            }
        }
        return false;
    }

    /* access modifiers changed from: protected */
    public void fieldName() {
        next();
        while (true) {
            char c = this.ch;
            if (c == '\\') {
                next();
                if (this.ch == 'u') {
                    next();
                    next();
                    next();
                    next();
                    next();
                } else {
                    next();
                }
            } else if (c == '\"') {
                next();
                return;
            } else {
                next();
            }
        }
    }

    /* access modifiers changed from: protected */
    public boolean string() {
        next();
        while (!this.eof) {
            char c = this.ch;
            if (c == '\\') {
                next();
                if (this.ch == 'u') {
                    next();
                    next();
                    next();
                    next();
                    next();
                } else {
                    next();
                }
            } else if (c == '\"') {
                next();
                return true;
            } else {
                next();
            }
        }
        return false;
    }

    /* access modifiers changed from: package-private */
    public void skipWhiteSpace() {
        while (isWhiteSpace(this.ch)) {
            next();
        }
    }

    static class UTF8Validator extends JSONValidator {
        private final byte[] bytes;

        public UTF8Validator(byte[] bArr) {
            this.bytes = bArr;
            next();
            skipWhiteSpace();
        }

        /* access modifiers changed from: package-private */
        @Override // com.alibaba.fastjson.JSONValidator
        public void next() {
            this.pos++;
            int i = this.pos;
            byte[] bArr = this.bytes;
            if (i >= bArr.length) {
                this.ch = 0;
                this.eof = true;
                return;
            }
            this.ch = (char) bArr[this.pos];
        }
    }

    static class UTF8InputStreamValidator extends JSONValidator {
        private static final ThreadLocal<byte[]> bufLocal = new ThreadLocal<>();
        private byte[] buf;
        private int end = -1;
        private final InputStream is;
        private int readCount = 0;

        public UTF8InputStreamValidator(InputStream inputStream) {
            this.is = inputStream;
            ThreadLocal<byte[]> threadLocal = bufLocal;
            byte[] bArr = threadLocal.get();
            this.buf = bArr;
            if (bArr != null) {
                threadLocal.set(null);
            } else {
                this.buf = new byte[8192];
            }
            next();
            skipWhiteSpace();
        }

        /* access modifiers changed from: package-private */
        @Override // com.alibaba.fastjson.JSONValidator
        public void next() {
            if (this.pos < this.end) {
                byte[] bArr = this.buf;
                int i = this.pos + 1;
                this.pos = i;
                this.ch = (char) bArr[i];
            } else if (!this.eof) {
                try {
                    InputStream inputStream = this.is;
                    byte[] bArr2 = this.buf;
                    int read = inputStream.read(bArr2, 0, bArr2.length);
                    this.readCount++;
                    if (read > 0) {
                        this.ch = (char) this.buf[0];
                        this.pos = 0;
                        this.end = read - 1;
                    } else if (read == -1) {
                        this.pos = 0;
                        this.end = 0;
                        this.buf = null;
                        this.ch = 0;
                        this.eof = true;
                    } else {
                        this.pos = 0;
                        this.end = 0;
                        this.buf = null;
                        this.ch = 0;
                        this.eof = true;
                        throw new JSONException("read error");
                    }
                } catch (IOException unused) {
                    throw new JSONException("read error");
                }
            }
        }

        @Override // java.io.Closeable, com.alibaba.fastjson.JSONValidator, java.lang.AutoCloseable
        public void close() throws IOException {
            bufLocal.set(this.buf);
            this.is.close();
        }
    }

    static class UTF16Validator extends JSONValidator {
        private final String str;

        public UTF16Validator(String str2) {
            this.str = str2;
            next();
            skipWhiteSpace();
        }

        /* access modifiers changed from: package-private */
        @Override // com.alibaba.fastjson.JSONValidator
        public void next() {
            this.pos++;
            if (this.pos >= this.str.length()) {
                this.ch = 0;
                this.eof = true;
                return;
            }
            this.ch = this.str.charAt(this.pos);
        }

        /* access modifiers changed from: protected */
        @Override // com.alibaba.fastjson.JSONValidator
        public final void fieldName() {
            char charAt;
            int i = this.pos;
            do {
                i++;
                if (i >= this.str.length() || (charAt = this.str.charAt(i)) == '\\') {
                    next();
                    while (true) {
                        if (this.ch == '\\') {
                            next();
                            if (this.ch == 'u') {
                                next();
                                next();
                                next();
                                next();
                                next();
                            } else {
                                next();
                            }
                        } else if (this.ch == '\"') {
                            next();
                            return;
                        } else if (!this.eof) {
                            next();
                        } else {
                            return;
                        }
                    }
                }
            } while (charAt != '\"');
            int i2 = i + 1;
            this.ch = this.str.charAt(i2);
            this.pos = i2;
        }
    }

    static class ReaderValidator extends JSONValidator {
        private static final ThreadLocal<char[]> bufLocal = new ThreadLocal<>();
        private char[] buf;
        private int end = -1;
        final Reader r;
        private int readCount = 0;

        ReaderValidator(Reader reader) {
            this.r = reader;
            ThreadLocal<char[]> threadLocal = bufLocal;
            char[] cArr = threadLocal.get();
            this.buf = cArr;
            if (cArr != null) {
                threadLocal.set(null);
            } else {
                this.buf = new char[8192];
            }
            next();
            skipWhiteSpace();
        }

        /* access modifiers changed from: package-private */
        @Override // com.alibaba.fastjson.JSONValidator
        public void next() {
            if (this.pos < this.end) {
                char[] cArr = this.buf;
                int i = this.pos + 1;
                this.pos = i;
                this.ch = cArr[i];
            } else if (!this.eof) {
                try {
                    Reader reader = this.r;
                    char[] cArr2 = this.buf;
                    int read = reader.read(cArr2, 0, cArr2.length);
                    this.readCount++;
                    if (read > 0) {
                        this.ch = this.buf[0];
                        this.pos = 0;
                        this.end = read - 1;
                    } else if (read == -1) {
                        this.pos = 0;
                        this.end = 0;
                        this.buf = null;
                        this.ch = 0;
                        this.eof = true;
                    } else {
                        this.pos = 0;
                        this.end = 0;
                        this.buf = null;
                        this.ch = 0;
                        this.eof = true;
                        throw new JSONException("read error");
                    }
                } catch (IOException unused) {
                    throw new JSONException("read error");
                }
            }
        }

        @Override // java.io.Closeable, com.alibaba.fastjson.JSONValidator, java.lang.AutoCloseable
        public void close() throws IOException {
            bufLocal.set(this.buf);
            this.r.close();
        }
    }
}
